<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Tour extends Model
{
    protected $fillable=[
        'title','tags', 'includes', 'excludes', 'slug','description','photo', 'location', 'map', 'age',
        'number_of_days', 'itinerary_titles', 'itinerary_descriptions', 'images', 'price', 'price_per_child',
        'cost_includes', 'cost_excludes', 'discount', 'tour_tag_id', 'include_tour_id', 'exclude_tour_id', 'location_category_id', 'added_by','status', 'views', 'availability_type', 'unavailable_periods'

    ];

    protected $casts = [
        'itinerary_titles' => 'array',
        'itinerary_descriptions' => 'array',
        'unavailable_periods' => 'array',
    ];

    public function tag_info(){
        return $this->hasOne('App\Models\TourTag','id','tour_tag_id');
    }

    public function author_info(){
        return $this->hasOne('App\User','id','added_by');
    }
    public static function getAllTour(){
        return Tour::with(['author_info'])->orderBy('id','ASC')->paginate(10);
    }

    public function include_info(){
        return $this->hasOne('App\Models\IncludeTour','id','include_tour_id');
    }

    public function exclude_info(){
        return $this->hasOne('App\Models\ExcludeTour','id','exclude_tour_id');
    }

    public function location_category(){
        return $this->belongsTo('App\Models\LocationCategory','location_category_id','id');
    }

    public static function getTourBySlug($slug){
        return Tour::with(['tag_info','author_info', 'include_info', 'exclude_info', 'location_category'])->where('slug',$slug)->first();
    }

    public static function getTourByTag($slug){
        // dd($slug);
        return Tour::where('tags',$slug)->paginate(30);
    }

    public static function countActiveTour(){
        $data=Tour::where('status','active')->count();
        if($data){
            return $data;
        }
        return 0;
    }

    public static function getMostViewedTours($limit = 4){
        return Tour::with(['tag_info','author_info', 'include_info', 'exclude_info', 'location_category'])
                   ->where('status','active')->orWhere('status','featured')
                   ->orderBy('views','DESC')
                   ->orderBy('id','DESC')
                   ->limit($limit)
                   ->get();
    }

    public function incrementViews(){
        $this->increment('views');
    }

    public static function getTourByCategory($slug, $tour_tag_id = null){
        $query = Tour::with(['tag_info','author_info', 'include_info', 'exclude_info', 'location_category'])->where('location_category_id',$slug)->orderBy('id','DESC');
        if ($tour_tag_id) {
            $query->where('tour_tag_id', $tour_tag_id);
        }
        return $query->get();
    }

    /**
     * Check if tour is available on a specific date
     */
    public function isAvailableOnDate($date = null)
    {
        if ($date === null) {
            $date = now();
        }
        
        if (is_string($date)) {
            $date = \Carbon\Carbon::parse($date);
        }

        // If tour is year-round available, return true
        if ($this->availability_type === 'year_round') {
            return true;
        }

        // Check unavailable periods
        if ($this->unavailable_periods && is_array($this->unavailable_periods)) {
            foreach ($this->unavailable_periods as $period) {
                $startDate = \Carbon\Carbon::parse($period['start_date']);
                $endDate = \Carbon\Carbon::parse($period['end_date']);
                
                if ($date->between($startDate, $endDate)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Get availability status text
     */
    public function getAvailabilityStatus()
    {
        switch ($this->availability_type) {
            case 'year_round':
                return 'Available Year Round';
            case 'seasonal':
                return 'Seasonal Availability';
            case 'custom':
                return 'Custom Availability';
            default:
                return 'Available';
        }
    }

    /**
     * Get next available period
     */
    public function getNextAvailablePeriod()
    {
        if ($this->availability_type === 'year_round') {
            return 'Available Now';
        }

        $today = now();
        
        if ($this->unavailable_periods && is_array($this->unavailable_periods)) {
            foreach ($this->unavailable_periods as $period) {
                $endDate = \Carbon\Carbon::parse($period['end_date']);
                
                if ($today->lessThan($endDate)) {
                    return 'Available from ' . $endDate->addDay()->format('M d, Y');
                }
            }
        }

        return 'Available Now';
    }

    /**
     * Scope to filter tours available during a specific date range
     */
    public static function getAvailableToursForDateRange($startDate, $endDate, $locationCategoryId = null, $tour_tag_id = null)
    {
        $query = static::with(['tag_info','author_info', 'include_info', 'exclude_info', 'location_category'])
            ->where(function($q) {
                $q->where('status', 'active')->orWhere('status', 'featured');
            });

        // Filter by location category if provided
        if ($locationCategoryId) {
            $query->where('location_category_id', $locationCategoryId);
        }
        if ($tour_tag_id) {
            $query->where('tour_tag_id', $tour_tag_id);
        }

        $tours = $query->get();

        // Filter out tours that are not available during the requested period
        $availableTours = $tours->filter(function($tour) use ($startDate, $endDate) {
            return $tour->isAvailableForDateRange($startDate, $endDate);
        });

        return $availableTours;
    }

    /**
     * Check if tour is available for an entire date range
     */
    public function isAvailableForDateRange($startDate, $endDate)
    {
        if (is_string($startDate)) {
            $startDate = \Carbon\Carbon::parse($startDate);
        }
        if (is_string($endDate)) {
            $endDate = \Carbon\Carbon::parse($endDate);
        }

        // If tour is year-round available, return true
        if ($this->availability_type === 'year_round') {
            return true;
        }

        // Check if any part of the requested date range overlaps with unavailable periods
        if ($this->unavailable_periods && is_array($this->unavailable_periods)) {
            foreach ($this->unavailable_periods as $period) {
                $unavailableStart = \Carbon\Carbon::parse($period['start_date']);
                $unavailableEnd = \Carbon\Carbon::parse($period['end_date']);
                
                // Check if there's any overlap between requested dates and unavailable period
                if ($startDate->lessThanOrEqualTo($unavailableEnd) && $endDate->greaterThanOrEqualTo($unavailableStart)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Get tours by category with availability filtering
     */
    public static function getTourByCategoryWithAvailability($categoryId, $startDate = null, $endDate = null, $tour_tag_id = null)
    {
        if ($startDate && $endDate) {
            return static::getAvailableToursForDateRange($startDate, $endDate, $categoryId, $tour_tag_id);
        }
        
        // Fallback to original method if no dates provided
        return static::getTourByCategory($categoryId, $tour_tag_id);
    }
     
}
