<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\Tour;
use App\Models\TourTag;
use App\Models\IncludeTour;
use App\Models\ExcludeTour;
use App\User;

class TourController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $tours = Tour::orderBy('id', 'ASC')->paginate(57);
        return view('backend.tour.index')->with('tours', $tours);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $tags = TourTag::get();
        $includes = IncludeTour::get();
        $excludes = ExcludeTour::get();
        $users = User::get();
        return view('backend.tour.create')
            ->with('users', $users)
            ->with('tags', $tags)
            ->with('includes', $includes)
            ->with('excludes', $excludes);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'string|required',
            'description' => 'string|nullable',
            'photo' => 'required|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'images.*' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'location' => 'string|required',
            'map' => 'string|nullable',
            'age' => 'string|required',
            'number_of_days' => 'string|required',
            'price' => 'string|required',
            'price_per_child' => 'string|nullable',
            'includetours' => 'nullable',
            'excludetours' => 'nullable',
            'discount' => 'string|nullable',
            'tags' => 'nullable',
            'includes' => 'nullable',
            'excludes' => 'nullable',
            'added_by' => 'nullable',
            'status' => 'required|in:active,inactive,featured',
            'itinerary_titles' => 'required|array',
            'itinerary_descriptions' => 'required|array',
            'availability_type' => 'required|in:year_round,seasonal,custom',
            'unavailable_periods' => 'nullable|array',
            'unavailable_periods.*.start_date' => 'nullable|date',
            'unavailable_periods.*.end_date' => 'nullable|date|after_or_equal:unavailable_periods.*.start_date',
            'availability_notes' => 'nullable|string',
        ]);

        $data = $request->all();

        $slug = Str::slug($request->title);
        $count = Tour::where('slug', $slug)->count();
        if ($count > 0) {
            $slug = $slug . '-' . date('ymdis') . '-' . rand(0, 999);
        }
        $data['slug'] = $slug;

        // Handle tags
        $tags = $request->input('tags',[]);
        if ($tags) {
            $tag_id = TourTag::where('title',$tags)->first();
            $data['tour_tag_id'] = $tag_id->id;
            $data['tags'] = $tags;
        } else {
            $data['tour_tag_id'] = '';
            $data['tags'] = '';
        }

        // Handle includes
        $includes = $request->input('includes');
        if ($includes) {
            $data['includes'] = implode(',', $includes);
        } else {
            $data['includes'] = '';
        }

        // Handle excludes
        $excludes = $request->input('excludes');
        if ($excludes) {
            $data['excludes'] = implode(',', $excludes);
        } else {
            $data['excludes'] = '';
        }

        // Handle photo upload
        $path = $request->file('photo')->store('public/images');
        $data['photo'] = $path;

        // Handle multiple image uploads
        $imagesPaths = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $imagesPaths[] = $image->store('public/multiple_images');
            }
        }
        $data['images'] = implode(',', $imagesPaths);

        // Handle itinerary titles and descriptions
        $data['itinerary_titles'] = json_encode($request->input('itinerary_titles'));
        $data['itinerary_descriptions'] = json_encode($request->input('itinerary_descriptions'));

        // Handle availability data
        $data['availability_type'] = $request->input('availability_type', 'year_round');
        
        // Process unavailable periods
        if ($request->has('unavailable_periods') && is_array($request->unavailable_periods)) {
            $unavailablePeriods = [];
            foreach ($request->unavailable_periods as $period) {
                if (!empty($period['start_date']) && !empty($period['end_date'])) {
                    $unavailablePeriods[] = [
                        'start_date' => $period['start_date'],
                        'end_date' => $period['end_date'],
                        'reason' => $period['reason'] ?? ''
                    ];
                }
            }
            $data['unavailable_periods'] = $unavailablePeriods;
        }

        $data['availability_notes'] = $request->input('availability_notes');

        $status = Tour::create($data);
        if ($status) {
            request()->session()->flash('success', 'Tour Successfully added');
        } else {
            request()->session()->flash('error', 'Please try again!!');
        }
        return redirect()->route('tour.index');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $tour = Tour::findOrFail($id);
        $tags = TourTag::get();
        $includes = IncludeTour::get();
        $excludes = ExcludeTour::get();
        $users = User::get();
        return view('backend.tour.edit')
            ->with('users', $users)
            ->with('tags', $tags)
            ->with('includes', $includes)
            ->with('excludes', $excludes)
            ->with('tour', $tour);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $tour = Tour::findOrFail($id);

        $this->validate($request, [
            'title' => 'string|required',
            'description' => 'string|nullable',
            'photo' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'images.*' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'location' => 'string|required',
            'map' => 'string|nullable',
            'age' => 'string|required',
            'number_of_days' => 'string|required',
            'price' => 'string|required',
            'price_per_child' => 'string|nullable',
            'discount' => 'string|nullable',
            'tags' => 'nullable',
            'includes' => 'nullable',
            'excludes' => 'nullable',
            'added_by' => 'nullable',
            'status' => 'required|in:active,inactive,featured',
            'itinerary_titles' => 'required|array',
            'itinerary_descriptions' => 'required|array',
            'availability_type' => 'required|in:year_round,seasonal,custom',
            'unavailable_periods' => 'nullable|array',
            'unavailable_periods.*.start_date' => 'nullable|date',
            'unavailable_periods.*.end_date' => 'nullable|date|after_or_equal:unavailable_periods.*.start_date',
        ]);

        $data = $request->all();

        // Handle tags
        $tags = $request->input('tags');
        if ($tags) {
            $tag_id = TourTag::where('title',$tags)->first();
            $data['tour_tag_id'] = $tag_id->id;
            $data['tags'] = $tags;
        } else {
            $data['tour_tag_id'] = '';
            $data['tags'] = '';
        }

        // Handle includes
        $includes = $request->input('includes');
        if ($includes) {
            $data['includes'] = implode(',', $includes);
        } else {
            $data['includes'] = '';
        }

        // Handle excludes
        $excludes = $request->input('excludes');
        if ($excludes) {
            $data['excludes'] = implode(',', $excludes);
        } else {
            $data['excludes'] = '';
        }

        // Handle photo update only if a new one is provided
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('public/images');
            $data['photo'] = $path;
        } else {
            unset($data['photo']);
        }

        // Handle multiple image updates
        if ($request->hasFile('images')) {
            $imagesPaths = [];
            foreach ($request->file('images') as $image) {
                $imagesPaths[] = $image->store('public/multiple_images');
            }
            $data['images'] = implode(',', $imagesPaths);
        }

        // Handle itinerary titles and descriptions
        $data['itinerary_titles'] = json_encode($request->input('itinerary_titles'));
        $data['itinerary_descriptions'] = json_encode($request->input('itinerary_descriptions'));

        // Handle availability data
        $data['availability_type'] = $request->input('availability_type', 'year_round');
        
        // Process unavailable periods
        if ($request->has('unavailable_periods') && is_array($request->unavailable_periods)) {
            $unavailablePeriods = [];
            foreach ($request->unavailable_periods as $period) {
                if (!empty($period['start_date']) && !empty($period['end_date'])) {
                    $unavailablePeriods[] = [
                        'start_date' => $period['start_date'],
                        'end_date' => $period['end_date'],
                        'reason' => $period['reason'] ?? ''
                    ];
                }
            }
            $data['unavailable_periods'] = $unavailablePeriods;
        }


        $status = $tour->fill($data)->save();

        if ($status) {
            request()->session()->flash('success', 'Tour successfully updated');
        } else {
            request()->session()->flash('error', 'Please try again!!');
        }

        return redirect()->route('tour.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $tour = Tour::findOrFail($id);

        $status = $tour->delete();

        if ($status) {
            request()->session()->flash('success', 'Tour successfully deleted');
        } else {
            request()->session()->flash('error', 'Error while deleting tour');
        }
        return redirect()->route('tour.index');
    }
}
