<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Team;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class TeamController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $teams = Team::orderBy('id', 'ASC')->paginate(57);
        return view('backend.team.index')->with('teams', $teams);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.team.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // return $request->all();
        $this->validate($request, [
            'name' => 'string|required|max:100',
            'position' => 'string|required|max:100',
            'photo' => 'required|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'status' => 'required|in:active,inactive',
        ]);
        
        $data=$request->all();
        $slug=Str::slug($request->title);
        $count=Team::where('slug',$slug)->count();
        if($count>0){
            $slug=$slug.'-'.date('ymdis').'-'.rand(0,999);
        }
        $data['slug']=$slug;
        $data['name'] = $request->get('name');
        $data['position'] = $request->get('position');
        $data['status'] = $request->get('status');

        // return $slug;
        $path = $request->file('photo')->store('public/images');
        $data['photo'] = $path; 
        $status=Team::create($data);
        if($status){
            request()->session()->flash('success','Team successfully added');
        }
        else{
            request()->session()->flash('error','Error occurred while adding team');
        }
        return redirect()->route('team.index');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $team=Team::findOrFail($id);
        return view('backend.team.edit')
            ->with('team',$team);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $team = Team::findOrFail($id);

        $this->validate($request, [
            'name' => 'string|required|max:100',
            'position' => 'string|required|max:100',
            'photo' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'status' => 'required|in:active,inactive',
        ]);        

        $data = $request->all();
        $data['name'] = $request->get('name');
        $data['position'] = $request->get('position');
        $data['status'] = $request->get('status');

        // Handle photo update if provided
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('public/images');
            $data['photo'] = $path;
        } else {
            unset($data['photo']); // Remove the 'photo' key if it is not provided in the request
        }

        $status = $team->fill($data)->save();

        if ($status) {
            request()->session()->flash('success', 'Team successfully updated');
        } else {
            request()->session()->flash('error', 'Error occurred while updating team');
        }

        return redirect()->route('team.index');
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $team=Team::findOrFail($id);
        $status=$team->delete();
        if($status){
            request()->session()->flash('success','Team successfully deleted');
        }
        else{
            request()->session()->flash('error','Error occurred while deleting team');
        }
        return redirect()->route('team.index');
    }

}
