<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Service;
use Illuminate\Support\Str;

class ServiceController extends Controller
{
    public function index()
    {
        $services = Service::orderBy('id', 'ASC')->paginate(57);
        return view('backend.service.index', compact('services'));
    }

    public function create()
    {
        return view('backend.service.create');
    }

    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'string|required',
            'subtitle' => 'string|required',
            'description' => 'string|nullable|max:300',
            'photo' => 'required|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'status' => 'required|in:active,inactive'
        ]);

        $data = $request->all();
        $slug = Str::slug($request->title);
        $count = Service::where('slug', $slug)->count();
        if ($count > 0) {
            $slug = $slug . '-' . date('ymdis') . '-' . rand(0, 999);
        }
        $data['slug'] = $slug;

        if ($request->hasFile('photo')) {
            $data['photo'] = $request->file('photo')->store('public/images');
        }

        $status = Service::create($data);
        if ($status) {
            request()->session()->flash('success', 'Service Successfully added');
        } else {
            request()->session()->flash('error', 'Please try again!!');
        }
        return redirect()->route('service.index');
    }

    public function edit($id)
    {
        $service = Service::findOrFail($id);
        return view('backend.service.edit', compact('service'));
    }

    public function update(Request $request, $id)
    {
        $service = Service::findOrFail($id);

        $this->validate($request, [
            'title' => 'string|required',
            'subtitle' => 'string|required',
            'description' => 'string|nullable|max:300',
            'photo' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'status' => 'required|in:active,inactive'
        ]);

        $data = $request->all();

        // Handle photo update only if a new one is provided
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('public/images');
            $data['photo'] = $path;
        } else {
            unset($data['photo']); // Remove the 'photo' key if it is not provided in the request
        }

        $status = $service->update($data);
        if ($status) {
            request()->session()->flash('success', 'Service Successfully updated');
        } else {
            request()->session()->flash('error', 'Please try again!!');
        }
        return redirect()->route('service.index');
    }

    public function destroy($id)
    {
        $service = Service::findOrFail($id);
        $status = $service->delete();
        if ($status) {
            request()->session()->flash('success', 'Service successfully deleted');
        } else {
            request()->session()->flash('error', 'Error while deleting service');
        }
        return redirect()->route('service.index');
    }
}
