<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PageBanner;
use Illuminate\Support\Str;

class PageBannerController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $pageBanners = PageBanner::orderBy('page_slug', 'ASC')->paginate(20);
        return view('backend.page-banner.index', compact('pageBanners'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.page-banner.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'page_slug' => 'required|string|max:255|unique:page_banners,page_slug',
            'banner_image' => 'required|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'status' => 'required|boolean'
        ]);

        $data = $request->all();
        
        // Handle file upload
        if ($request->hasFile('banner_image')) {
            $path = $request->file('banner_image')->store('public/page-banners');
            $data['banner_image'] = $path;
        }

        $status = PageBanner::create($data);
        
        if ($status) {
            request()->session()->flash('success', 'Page Banner successfully created');
        } else {
            request()->session()->flash('error', 'Error occurred while creating page banner');
        }
        
        return redirect()->route('page-banner.index');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $pageBanner = PageBanner::findOrFail($id);
        return view('backend.page-banner.show', compact('pageBanner'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $pageBanner = PageBanner::findOrFail($id);
        return view('backend.page-banner.edit', compact('pageBanner'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $pageBanner = PageBanner::findOrFail($id);
        
        $this->validate($request, [
            'page_slug' => 'required|string|max:255|unique:page_banners,page_slug,' . $id,
            'banner_image' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'status' => 'required|boolean'
        ]);

        $data = $request->all();
        
        // Handle file upload
        if ($request->hasFile('banner_image')) {
            // Delete old image if exists
            if ($pageBanner->banner_image && file_exists(public_path('storage/' . $pageBanner->banner_image))) {
                unlink(public_path('storage/' . $pageBanner->banner_image));
            }
            
            $path = $request->file('banner_image')->store('public/page-banners');
            $data['banner_image'] = $path;
        } else {
            unset($data['banner_image']); // Keep existing image
        }

        $status = $pageBanner->fill($data)->save();
        
        if ($status) {
            request()->session()->flash('success', 'Page Banner successfully updated');
        } else {
            request()->session()->flash('error', 'Error occurred while updating page banner');
        }
        
        return redirect()->route('page-banner.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $pageBanner = PageBanner::findOrFail($id);
        
        // Delete image file if exists
        if ($pageBanner->banner_image && file_exists(public_path('storage/' . $pageBanner->banner_image))) {
            unlink(public_path('storage/' . $pageBanner->banner_image));
        }
        
        $status = $pageBanner->delete();
        
        if ($status) {
            request()->session()->flash('success', 'Page Banner successfully deleted');
        } else {
            request()->session()->flash('error', 'Error occurred while deleting page banner');
        }
        
        return redirect()->route('page-banner.index');
    }
}
