<?php

namespace App\Http\Controllers;
use Auth;
use Illuminate\Http\Request;
use App\Models\Message;
use App\Events\MessageSent;
class MessageController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(){
        $messages=Message::paginate(20);
        return view('backend.message.index')->with('messages',$messages);
    }
    public function messageFive()
    {
        $message=Message::whereNull('read_at')->limit(5)->get();
        return response()->json($message);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request,[
            'name'=>'string|required|min:2',
            'email'=>'email|required',
            'message'=>'required|min:50|max:1000 ',
            'subject'=>'string|required',
            'cf-turnstile-response'=>'required'
        ]);

        // Verify Turnstile token
        $turnstileResponse = $request->input('cf-turnstile-response');
        if (!$this->verifyTurnstile($turnstileResponse)) {
            request()->session()->flash('error','Security verification failed. Please try again.');
            return redirect()->route('contact');
        }

        // return $request->all();

        $message=Message::create($request->all());
            // return $message;
        $data=array();
        $data['url']=route('message.show',$message->id);
        $data['date']=$message->created_at->format('F d, Y h:i A');
        $data['name']=$message->name;
        $data['email']=$message->email;
        $data['message']=$message->message;
        $data['subject']=$message->subject;

        if($message){
            request()->session()->flash('success','Message Successfully sent');
        }
        else{
            request()->session()->flash('error','Please try again!!');
        }
        return redirect()->route('contact');
    }

    /**
     * Verify Cloudflare Turnstile token
     *
     * @param string $token
     * @return bool
     */
    private function verifyTurnstile($token)
    {
        $secretKey = env('TURNSTILE_SECRET_KEY');
        
        if (!$secretKey) {
            // If no secret key is configured, skip verification in development
            return true;
        }

        $data = [
            'secret' => $secretKey,
            'response' => $token,
            'remoteip' => request()->ip()
        ];

        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data)
            ]
        ];

        $context = stream_context_create($options);
        $result = file_get_contents('https://challenges.cloudflare.com/turnstile/v0/siteverify', false, $context);
        
        if ($result === false) {
            return false;
        }

        $response = json_decode($result, true);
        return isset($response['success']) && $response['success'] === true;
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request,$id)
    {
        $message=Message::find($id);
        if($message){
            $message->read_at=\Carbon\Carbon::now();
            $message->save();
            return view('backend.message.show')->with('message',$message);
        }
        else{
            return back();
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $message=Message::find($id);
        $status=$message->delete();
        if($status){
            request()->session()->flash('success','Successfully deleted message');
        }
        else{
            request()->session()->flash('error','Error occurred please try again');
        }
        return back();
    }
}