<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\LocationCategory;
use Illuminate\Support\Facades\Storage;

class LocationCategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $location_categories = LocationCategory::orderBy('id', 'ASC')->paginate(10);
        return view('backend.location-category.index')->with('location_categories', $location_categories);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.location-category.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'string|required|max:200',
            'description' => 'string|nullable',
            'photo' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg',
            'banner' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg',
            'status' => 'required|in:active,inactive',
        ]);

        $data = $request->all();
        $slug = Str::slug($request->title);
        // Handle photo update only if a new one is provided
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('public/images');
            $data['photo'] = $path;
        } else {
            unset($data['photo']); // Remove the 'photo' key if it is not provided in the request
        }
        
        // Handle banner update only if a new one is provided
        if ($request->hasFile('banner')) {
            $path = $request->file('banner')->store('public/images');
            $data['banner'] = $path;
        } else {
            unset($data['banner']); // Remove the 'banner' key if it is not provided in the request
        }
        $count = LocationCategory::where('slug', $slug)->count();
        if ($count > 0) {
            $slug = $slug . '-' . date('ymdis') . '-' . rand(0, 999);
        }
        $data['slug'] = $slug;

        $status = LocationCategory::create($data);
        if ($status) {
            request()->session()->flash('success', 'Location Category successfully added');
        } else {
            request()->session()->flash('error', 'Please try again!!');
        }
        return redirect()->route('location-category.index');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $location_category = LocationCategory::findOrFail($id);
        return view('backend.location-category.show')->with('location_category', $location_category);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $location_category = LocationCategory::findOrFail($id);
        return view('backend.location-category.edit')->with('location_category', $location_category);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $location_category = LocationCategory::findOrFail($id);
        $this->validate($request, [
            'title' => 'string|required|max:200',
            'description' => 'string|nullable',
            'photo' => 'image|nullable|mimes:jpg,png,jpeg,gif,svg',
            'banner' => 'image|nullable|mimes:jpg,png,jpeg,gif,svg',
            'status' => 'required|in:active,inactive',
        ]);

        $data = $request->all();
        // Handle photo update only if a new one is provided
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('public/images');
            $data['photo'] = $path;
        } else {
            unset($data['photo']); // Remove the 'photo' key if it is not provided in the request
        }
        
        // Handle banner update only if a new one is provided
        if ($request->hasFile('banner')) {
            $path = $request->file('banner')->store('public/images');
            $data['banner'] = $path;
        } else {
            unset($data['banner']); // Remove the 'banner' key if it is not provided in the request
        }
        $status = $location_category->fill($data)->save();
        if ($status) {
            request()->session()->flash('success', 'Location Category successfully updated');
        } else {
            request()->session()->flash('error', 'Please try again!!');
        }
        return redirect()->route('location-category.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $location_category = LocationCategory::findOrFail($id);
        
        // Check if there are tours or destinations using this category
        $tours_count = $location_category->tours()->count();
        $destinations_count = $location_category->destinations()->count();
        
        if ($tours_count > 0 || $destinations_count > 0) {
            request()->session()->flash('error', 'Cannot delete location category. It is being used by ' . $tours_count . ' tours and ' . $destinations_count . ' destinations.');
            return redirect()->route('location-category.index');
        }

        // Delete photo if exists
        if ($location_category->photo) {
            Storage::delete('public/' . $location_category->photo);
        }

        $status = $location_category->delete();

        if ($status) {
            request()->session()->flash('success', 'Location Category successfully deleted');
        } else {
            request()->session()->flash('error', 'Error while deleting location category');
        }
        return redirect()->route('location-category.index');
    }

    public function getLocationCategoryByStatus($status, $id = null)
    {
        if ($id == null) {
            $location_categories = LocationCategory::select('title', 'id')->where('status', $status)->get();
        } else {
            $location_categories = LocationCategory::select('title', 'id')->where('id', $id)->where('status', $status)->get();
        }
        return $location_categories;
    }
}
