<?php

namespace App\Http\Controllers;
use App\Models\Banner;
use App\Models\PageBanner;
use App\Models\PostTag;
use App\Models\Post;
use App\Models\Team;
use App\Models\FAQ;
use App\Models\Testimonial;
use App\Models\Document;
use App\Models\Service;
use App\Models\Statistic;
use App\Models\Destination;
use App\Models\DestinationTag;
use App\Models\Tour;
use App\Models\TourTag;
use App\Models\Video;
use App\Models\Category;
use App\Models\LocationCategory;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;

class FrontendController extends Controller
{
    public function lang($locale)
    {
        App::setLocale($locale);
        session()->put('locale', $locale);
        return redirect()->back();
    }
   
    public function index(Request $request){
        return redirect()->route($request->user()->role);
    }

    public function home(){
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $banners=Banner::where('status','active')->limit(10)->orderBy('id','DESC')->get();
        $testimonials=Testimonial::where('status','active')->limit(3)->orderBy('id','DESC')->get();
        $teams=Team::where('status','active')->limit(3)->orderBy('id','DESC')->get();
        $statistics=Statistic::where('status','active')->orderBy('id','DESC')->limit(4)->get();
        $posts=Post::where('status','active')->limit(3)->orderBy('id','DESC')->get();
        $services=Service::where('status','active')->orderBy('id','DESC')->limit(6)->get();
        $destinations=Destination::where('status','featured')->orderBy('id','DESC')->limit(15)->get();
        $tours=Tour::where('status','featured')->orderBy('id','DESC')->limit(3)->get();
        $trending_tours=Tour::getMostViewedTours(4); // Get most viewed tours for trending section
        $categories=Category::where('status','active')->orderBy('id','DESC')->limit(4)->get();
        $countries = Destination::select('location')->distinct()->get()->pluck('location');
        $tags = Destination::select('tags')->distinct()->get()->pluck('tags');
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        $tour_tags=TourTag::where('status','active')->orderBy('id','DESC')->get();
        // return $category;
        return view('frontend.index')
                ->with('recent_posts',$recent_posts)
                ->with('banners',$banners)
                ->with('testimonials',$testimonials)
                ->with('teams',$teams)
                ->with('statistics',$statistics)
                ->with('posts',$posts)
                ->with('services',$services)
                ->with('destinations',$destinations)
                ->with('tours',$tours)
                ->with('trending_tours',$trending_tours)
                ->with('categories',$categories)
                ->with('countries',$countries)
                ->with('tags',$tags)
                ->with('location_categories',$location_categories)
                ->with('tour_tags',$tour_tags);
                
    }   
    public function tourSearch(Request $request){
        // Store original request data
        $originalRequest = $request->all();
        //check for type 
        $location_category = LocationCategory::where('id', $request->location)->first();
        $tour_tag = TourTag::where('id', $request->tour_type)->first();
        $data['slug'] = $location_category->slug;
        
        // Parse date range if provided
        $startDate = null;
        $endDate = null;
        
        if ($request->has('duration') && !empty($request->duration)) {
            // Parse the date range (format: "08-31 to 09-29" or similar)
            $dateRange = $request->duration;
            $currentYear = date('Y');
            
            // Handle different date formats
            if (strpos($dateRange, ' to ') !== false) {
                $dates = explode(' to ', $dateRange);
                $startDateRaw = trim($dates[0]);
                $endDateRaw = isset($dates[1]) ? trim($dates[1]) : $startDateRaw;
                
                // Convert MM-DD format to YYYY-MM-DD
                $startDate = $this->formatDateWithYear($startDateRaw, $currentYear);
                $endDate = $this->formatDateWithYear($endDateRaw, $currentYear);
                
                // Handle year rollover (if end date is earlier in year than start date, assume next year)
                if ($endDate < $startDate) {
                    $endDate = $this->formatDateWithYear($endDateRaw, $currentYear + 1);
                }
                
            } elseif (strpos($dateRange, ' - ') !== false) {
                $dates = explode(' - ', $dateRange);
                $startDateRaw = trim($dates[0]);
                $endDateRaw = isset($dates[1]) ? trim($dates[1]) : $startDateRaw;
                
                // Convert MM-DD format to YYYY-MM-DD
                $startDate = $this->formatDateWithYear($startDateRaw, $currentYear);
                $endDate = $this->formatDateWithYear($endDateRaw, $currentYear);
                
                // Handle year rollover
                if ($endDate < $startDate) {
                    $endDate = $this->formatDateWithYear($endDateRaw, $currentYear + 1);
                }
                
            } else {
                // Single date provided
                $startDate = $this->formatDateWithYear($dateRange, $currentYear);
                $endDate = $startDate;
            }
        }
        
        // convert to request object for category routing
        $categoryRequest = new Request($data);
            // Handle safari tours with availability filtering
            try {
                return $this->tourByCategoryWithAvailability($location_category->id, $startDate, $endDate, $tour_tag->id);
            } catch (\Exception $e) {
                // Fallback to regular category search if availability filtering fails
                return self::tourByCategory($categoryRequest);
            }
    }
    
    public function aboutUs(){
        $testimonial=Testimonial::where('status','active')->orderBy('id','DESC')->get();
        $teams=Team::where('status','active')->orderBy('id','DESC')->get();
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        $tour_tags=TourTag::where('status','active')->orderBy('id','DESC')->get();
        $banner=PageBanner::where('status',true)->where('page_slug','about-us')->first();

        return view('frontend.pages.about-us')
        ->with('testimonials',$testimonial)
            ->with('teams',$teams)
            ->with('banner',$banner)
            ->with('recent_posts',$recent_posts)
            ->with('location_categories',$location_categories)
            ->with('tour_tags',$tour_tags);    
    }

    public function whatweAre(){
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        $tour_tags=TourTag::where('status','active')->orderBy('id','DESC')->get();
        return view('frontend.pages.what-we-are')
            ->with('recent_posts',$recent_posts)
            ->with('tour_tags',$tour_tags)
            ->with('location_categories',$location_categories);
    }

    public function faq(){
        $faq=FAQ::where('status','active')->orderBy('id','DESC')->get();
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        $tour_tags=TourTag::where('status','active')->orderBy('id','DESC')->get();
        $banner=PageBanner::where('status',true)->where('page_slug','faq')->first();
        return view('frontend.pages.faq')
            ->with('faqs',$faq)
            ->with('recent_posts',$recent_posts)
            ->with('location_categories',$location_categories)
            ->with('tour_tags',$tour_tags)
            ->with('banner',$banner);
    }

    public function pricing(){
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        return view('frontend.pages.pricing')
            ->with('recent_posts',$recent_posts)
            ->with('location_categories',$location_categories);
    }

    public function statistic(){
        $statistic=Statistic::where('status','active')->orderBy('id','DESC')->get();
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        return view('frontend.pages.statistic')
            ->with('recent_posts',$recent_posts)
            ->with('location_categories',$location_categories)
            ->with('statistics',$statistic);
    }

    public function testimonial(){
        $testimonial=Testimonial::where('status','active')->orderBy('id','DESC')->get();
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        $banner=PageBanner::where('status',true)->where('page_slug','testimonial')->first();
        return view('frontend.pages.testimonial')
            ->with('recent_posts',$recent_posts)
            ->with('location_categories',$location_categories)
            ->with('testimonials',$testimonial)
            ->with('banner',$banner);
    }

    public function document(){
        $document=Document::where('status','active')->orderBy('id','DESC')->get();
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        $banner=PageBanner::where('status',true)->where('page_slug','document')->first();
        return view('frontend.pages.document')
            ->with('recent_posts',$recent_posts)
            ->with('location_categories',$location_categories)
            ->with('documents',$document)
            ->with('banner',$banner);
    }

    public function downloadFile($id){
        $document = Document::find($id);
        $path = Document::where("id", $id)->value("name");
        return asset('/storage/' . $document->name);
        $path = storage_path('public/' . $document->location);
        return Storage::download($path);
    }

    public function video(){
        $video=Video::where('status','active')->orderBy('id','DESC')->get();
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        $banner=PageBanner::where('status',true)->where('page_slug','video')->first();
        return view('frontend.pages.video')
            ->with('recent_posts',$recent_posts)
            ->with('videos',$video)
            ->with('banner',$banner);
    }

    public function contact(){
        $recent_posts=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        $banner=PageBanner::where('status',true)->where('page_slug','contact')->first();
        return view('frontend.pages.contact')
            ->with('recent_posts',$recent_posts)
            ->with('banner',$banner);
    }

    //News

    public function blog(){
        $post=Post::query();
        
        if(!empty($_GET['tag'])){
            $slug=explode(',',$_GET['tag']);
            // dd($slug);
            $tag_ids=PostTag::select('id')->whereIn('slug',$slug)->pluck('id')->toArray();
            // return $tag_ids;
            $post->where('post_tag_id',$tag_ids);
            // return $post;
        }

        if(!empty($_GET['show'])){
            $post=$post->where('status','active')->orderBy('id','DESC')->paginate($_GET['show']);
        }
        else{
            $post=$post->where('status','active')->orderBy('id','DESC')->paginate(500);
        }
        // $post=Post::where('status','active')->paginate(8);
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        $banner=PageBanner::where('status',true)->where('page_slug','blog')->first();
        return view('frontend.pages.blog')
        ->with('posts',$post)
        ->with('recent_posts',$rcnt_post)
        ->with('location_categories',$location_categories)
        ->with('banner',$banner);
    }

    public function blogDetail($slug){
        $post=Post::getPostBySlug($slug);
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $tags = Post::uniqueTags();
        $alltags = PostTag::where('status','active')->orderBy('id','DESC')->get();
        $banner=PageBanner::where('status',true)->where('page_slug','blog-detail')->first();

        return view('frontend.pages.blog-detail')->with('post',$post)->with('recent_posts',$rcnt_post)->with('alltags',$alltags)->with('banner',$banner);
    }

    public function blogSearch(Request $request){
        // return $request->all();
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $posts=Post::orwhere('title','like','%'.$request->search.'%')
            ->orwhere('description','like','%'.$request->search.'%')
            ->orwhere('slug','like','%'.$request->search.'%')
            ->orderBy('id','DESC')
            ->paginate(800);
        $banner=PageBanner::where('status',true)->where('page_slug','blog')->first();
        return view('frontend.pages.blog')->with('posts',$posts)->with('recent_posts',$rcnt_post)->with('banner',$banner);
    }

    public function blogFilter(Request $request){
        $data=$request->all();
        // return $data;
        $tagURL="";
        if(!empty($data['tag'])){
            foreach($data['tag'] as $tag){
                if(empty($tagURL)){
                    $tagURL .='&tag='.$tag;
                }
                else{
                    $tagURL .=','.$tag;
                }
            }
        }
        // return $tagURL;
        return redirect()->route('blog',$tagURL);
    }
    public function blogByTag(Request $request){
        // dd($request->slug);
        $post=Post::getBlogByTag($request->slug);
        // return $post;
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $banner=PageBanner::where('status',true)->where('page_slug','blog')->first();
        return view('frontend.pages.blog')->with('posts',$post)->with('recent_posts',$rcnt_post)->with('banner',$banner);
    }

    //Destination

    public function destination(){
        $destination=Destination::query();

        if(!empty($_GET['tag'])){
            $slug=explode(',',$_GET['tag']);
            // dd($slug);
            $tag_ids=DestinationTag::select('id')->whereIn('slug',$slug)->pluck('id')->toArray();
            // return $tag_ids;
            $destination->where('destination_tag_id',$tag_ids);
            // return $destination;
        }

        if(!empty($_GET['show'])){
            $destination=$destination->where('status','active')->orderBy('id','DESC')->paginate($_GET['show']);
        }
        else{
            $destination=$destination->where('status','active')->orderBy('id','DESC')->paginate(500);
        }
        // $post=Post::where('status','active')->paginate(8);
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        return view('frontend.pages.destination')
        ->with('destinations',$destination)
        ->with('recent_posts',$rcnt_post);
    }

    public function destinationByTag(Request $request){
        // dd($request->slug);
        $destination=Destination::getDestinationByTag($request->slug);
        // return $post;
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        return view('frontend.pages.destination')->with('destinations',$destination)->with('recent_posts',$rcnt_post);
    }

    public function destinationDetail($slug){
        $destination=Destination::getDestinationBySlug($slug);
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $location_category = LocationCategory::where('id',$destination->location_category_id)->first();
        // return $post;
        return view('frontend.pages.destination-detail')->with('destination',$destination)
            ->with('recent_posts',$rcnt_post)
            ->with('location_category',$location_category);
    }
    
    public function destinationSearch(Request $request) {
        $query = Destination::query();
    
        if ($request->has('location') && $request->location != '') {
            $query->where('title', 'like', '%' . $request->location . '%')
                  ->orWhere('location', 'like', '%' . $request->location . '%');
        }
    
        if ($request->has('tags') && $request->tags != '') {
            $query->where('tags', 'like', '%' . $request->tags . '%');
        }
    
        $destinations = $query->where('status', 'active')->orderBy('id', 'DESC')->paginate(800);
        
    
        return view('frontend.pages.search_results', compact('destinations'));
    }

    public function destinationAutocomplete(Request $request) {
        $query = $request->get('q', '');
        
        // Return empty array if query is too short
        if (strlen($query) < 2) {
            return response()->json([]);
        }
        
        // Clean the query
        $query = trim($query);
        
        // Search in both title and location fields
        $destinations = Destination::where(function($q) {
                $q->where('status', 'active')
                  ->orWhere('status', 'featured');
            })
            ->where(function($q) use ($query) {
                $q->where('title', 'like', '%' . $query . '%')
                  ->orWhere('location', 'like', '%' . $query . '%')
                  ->orWhere('description', 'like', '%' . $query . '%');
            })
            ->select('id', 'title', 'location', 'slug')
            ->orderBy('title', 'asc')
            ->limit(10)
            ->get();
        
        $results = $destinations->map(function($destination) {
            return [
                'id' => $destination->id,
                'title' => $destination->title,
                'location' => $destination->location,
                'slug' => $destination->slug,
                'display_text' => $destination->title . ' - ' . $destination->location
            ];
        });
        
        return response()->json($results);
    }
    

    public function show($slug) {
        $destination = Destination::where('slug', $slug)->firstOrFail();
        return view('frontend.pages.destination-detail', compact('destination'));
    }    
        
    //Tour

    public function tour(){
        $tour=Tour::query();

        if(!empty($_GET['tag'])){
            $slug=explode(',',$_GET['tag']);
            // dd($slug);
            $tag_ids=TourTag::select('id')->whereIn('slug',$slug)->pluck('id')->toArray();
            // return $tag_ids;
            $tour->where('tour_tag_id',$tag_ids);
            // return $tour;
        }

        if(!empty($_GET['show'])){
            $tour=$tour->where('status','active')->orWhere('status','featured')->orderBy('id','DESC')->paginate($_GET['show']);
        }
        else{
            $tour=$tour->where('status','active')->orWhere('status','featured')->orderBy('id','DESC')->paginate(500);
        }
        // $post=Post::where('status','active')->paginate(8);
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        return view('frontend.pages.tour')
        ->with('tours',$tour)
        ->with('recent_posts',$rcnt_post);
    }

    public function tourByTag(Request $request){
        // dd($request->slug);
        $tourTag = TourTag::where('slug',$request->slug)->first();
        $tour=Tour::getTourByTag($request->slug);
        // return $post;
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        return view('frontend.pages.tour')->with('tours',$tour)->with('recent_posts',$rcnt_post)->with('tourtag',$tourTag);
    }

    public function tourDetail($slug){
        $tour=Tour::getTourBySlug($slug);
        if($tour) {
            $tour->incrementViews(); // Increment view count
        }
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        $recent_tours=Tour::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        // return $post;
        return view('frontend.pages.tour-detail')->with('tour',$tour)
            ->with('recent_posts',$rcnt_post)
            ->with('recent_tours',$recent_tours);
    }
    public function tourByCategory(Request $request){
        // dd($request->slug);
        $location_category = LocationCategory::where('slug',$request->slug)->first();
        $tour=Tour::getTourByCategory($location_category->id);
        // return $post;
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        return view('frontend.pages.tour')->with('tours',$tour)->with('recent_posts',$rcnt_post);
    }

    public function tourByCategoryWithAvailability($categoryId, $startDate = null, $endDate = null, $tour_tag_id = null){
        $tours = Tour::getTourByCategoryWithAvailability($categoryId, $startDate, $endDate, $tour_tag_id);
        $rcnt_post = Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        
        // Add search context for the view
        $searchContext = [
            'location_category_id' => $categoryId,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'has_date_filter' => !is_null($startDate) && !is_null($endDate)
        ];
        
        return view('frontend.pages.tour')
            ->with('tours', $tours)
            ->with('recent_posts', $rcnt_post)
            ->with('search_context', $searchContext);
    }

    public function tourFilter(Request $request){
        $data = $request->all();
        
        // Get the location category
        $locationCategoryId = $data['location'] ?? null;
        $startDate = null;
        $endDate = null;
        
        // Parse date range if provided
        if (isset($data['duration']) && !empty($data['duration'])) {
            $dateRange = $data['duration'];
            $currentYear = date('Y');
            
            // Handle different date formats
            if (strpos($dateRange, ' to ') !== false) {
                $dates = explode(' to ', $dateRange);
                $startDateRaw = trim($dates[0]);
                $endDateRaw = isset($dates[1]) ? trim($dates[1]) : $startDateRaw;
                
                // Convert MM-DD format to YYYY-MM-DD
                $startDate = $this->formatDateWithYear($startDateRaw, $currentYear);
                $endDate = $this->formatDateWithYear($endDateRaw, $currentYear);
                
                // Handle year rollover (if end date is earlier in year than start date, assume next year)
                if ($endDate < $startDate) {
                    $endDate = $this->formatDateWithYear($endDateRaw, $currentYear + 1);
                }
                
            } elseif (strpos($dateRange, ' - ') !== false) {
                $dates = explode(' - ', $dateRange);
                $startDateRaw = trim($dates[0]);
                $endDateRaw = isset($dates[1]) ? trim($dates[1]) : $startDateRaw;
                
                // Convert MM-DD format to YYYY-MM-DD
                $startDate = $this->formatDateWithYear($startDateRaw, $currentYear);
                $endDate = $this->formatDateWithYear($endDateRaw, $currentYear);
                
                // Handle year rollover
                if ($endDate < $startDate) {
                    $endDate = $this->formatDateWithYear($endDateRaw, $currentYear + 1);
                }
                
            } else {
                // Single date provided
                $startDate = $this->formatDateWithYear($dateRange, $currentYear);
                $endDate = $startDate;
            }
        }
        
        // Get filtered tours
        if ($locationCategoryId) {
            $tours = Tour::getTourByCategoryWithAvailability($locationCategoryId, $startDate, $endDate);
        } else {
            $tours = Tour::getAvailableToursForDateRange($startDate, $endDate);
        }
        
        $rcnt_post = Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        
        // Add search context for the view
        $searchContext = [
            'location_category_id' => $locationCategoryId,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'has_date_filter' => !is_null($startDate) && !is_null($endDate),
            'search_performed' => true
        ];
        
        return view('frontend.pages.tour')
            ->with('tours', $tours)
            ->with('recent_posts', $rcnt_post)
            ->with('search_context', $searchContext);
    }

    /**
     * Helper method to format MM-DD date with year
     */
    private function formatDateWithYear($dateString, $year)
    {
        // Handle MM-DD format (e.g., "08-31")
        if (preg_match('/^(\d{1,2})-(\d{1,2})$/', $dateString, $matches)) {
            $month = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
            $day = str_pad($matches[2], 2, '0', STR_PAD_LEFT);
            
            // Validate the date
            if (checkdate($month, $day, $year)) {
                return $year . '-' . $month . '-' . $day;
            }
        }
        
        // Handle YYYY-MM-DD format (already formatted)
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateString)) {
            return $dateString;
        }
        
        // Handle other formats, try to parse with Carbon
        try {
            $carbonDate = \Carbon\Carbon::parse($dateString);
            return $carbonDate->format('Y-m-d');
        } catch (\Exception $e) {
            // If all else fails, return current date
            return date('Y-m-d');
        }
    }

    public function destinationByCategory(Request $request){
        // dd($request->slug);
        $location_category = LocationCategory::where('slug',$request->slug)->first();
        $destination=Destination::getDestinationByCategory($location_category->id);
        $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
        // return $post;
        $rcnt_post=Post::where('status','active')->orderBy('id','DESC')->limit(3)->get();
        return view('frontend.pages.destination')->with('destinations',$destination)->with('recent_posts',$rcnt_post)
        ->with('location_category',$location_category)
        ->with('location_categories',$location_categories);
    }


    public function book(){
        $tours=Tour::get();
         $tags = TourTag::get();
         $location_categories=LocationCategory::where('status','active')->orderBy('id','DESC')->get();
         $banner=PageBanner::where('status',true)->where('page_slug','book')->first();
        return view('frontend.pages.book',compact('tours','tags','location_categories','banner'));
    }

    public function searchAll(Request $request) {
        $query = $request->get('q', '');
        $results = [];
        
        if (strlen($query) >= 2) {
            // Search Tours
            $tours = Tour::where(function($q) {
                    $q->where('status', 'active')->orWhere('status', 'featured');
                })
                ->where(function($q) use ($query) {
                    $q->where('title', 'like', '%' . $query . '%')
                      ->orWhere('location', 'like', '%' . $query . '%')
                      ->orWhere('description', 'like', '%' . $query . '%');
                })
                ->select('id', 'title', 'location', 'slug', 'photo')
                ->limit(5)
                ->get()
                ->map(function($tour) {
                    return [
                        'id' => $tour->id,
                        'title' => $tour->title,
                        'location' => $tour->location,
                        'slug' => $tour->slug,
                        'photo' => $tour->photo,
                        'type' => 'tour',
                        'url' => route('tour.detail', $tour->slug)
                    ];
                });

            // Search Destinations
            $destinations = Destination::where(function($q) {
                    $q->where('status', 'active')->orWhere('status', 'featured');
                })
                ->where(function($q) use ($query) {
                    $q->where('title', 'like', '%' . $query . '%')
                      ->orWhere('location', 'like', '%' . $query . '%')
                      ->orWhere('description', 'like', '%' . $query . '%');
                })
                ->select('id', 'title', 'location', 'slug', 'photo')
                ->limit(5)
                ->get()
                ->map(function($destination) {
                    return [
                        'id' => $destination->id,
                        'title' => $destination->title,
                        'location' => $destination->location,
                        'slug' => $destination->slug,
                        'photo' => $destination->photo,
                        'type' => 'destination',
                        'url' => route('destination.detail', $destination->slug)
                    ];
                });

            // Search Articles/Posts
            $posts = Post::where('status', 'active')
                ->where(function($q) use ($query) {
                    $q->where('title', 'like', '%' . $query . '%')
                      ->orWhere('description', 'like', '%' . $query . '%')
                      ->orWhere('quote', 'like', '%' . $query . '%')
                      ->orWhere('summary', 'like', '%' . $query . '%');
                })
                ->select('id', 'title', 'slug', 'photo', 'quote')
                ->limit(5)
                ->get()
                ->map(function($post) {
                    return [
                        'id' => $post->id,
                        'title' => $post->title,
                        'slug' => $post->slug,
                        'photo' => $post->photo,
                        'quote' => $post->quote,
                        'type' => 'article',
                        'url' => route('blog.detail', $post->slug)
                    ];
                });

            $results = [
                'tours' => $tours,
                'destinations' => $destinations,
                'posts' => $posts,
                'total' => $tours->count() + $destinations->count() + $posts->count()
            ];
        }

        $banner=PageBanner::where('status',true)->where('page_slug','search')->first();
        return view('frontend.pages.search-results', compact('results', 'query', 'banner'));
    }

    public function globalAutocomplete(Request $request) {
        $query = $request->get('q', '');
        if (strlen($query) < 2) {
            return response()->json([]);
        }
        
        $query = trim($query);
        $results = [];
        
        // Search Tours
        $tours = Tour::where(function($q) {
                $q->where('status', 'active')->orWhere('status', 'featured');
            })
            ->where(function($q) use ($query) {
                $q->where('title', 'like', '%' . $query . '%')
                  ->orWhere('location', 'like', '%' . $query . '%');
            })
            ->select('id', 'title', 'location', 'slug')
            ->limit(3)
            ->get()
            ->map(function($tour) {
                return [
                    'id' => $tour->id,
                    'title' => $tour->title,
                    'location' => $tour->location,
                    'slug' => $tour->slug,
                    'type' => 'tour',
                    'display_text' => $tour->title . ' - ' . $tour->location,
                    'url' => route('tour.detail', $tour->slug)
                ];
            });

        // Search Destinations
        $destinations = Destination::where(function($q) {
                $q->where('status', 'active')->orWhere('status', 'featured');
            })
            ->where(function($q) use ($query) {
                $q->where('title', 'like', '%' . $query . '%')
                  ->orWhere('location', 'like', '%' . $query . '%');
            })
            ->select('id', 'title', 'location', 'slug')
            ->limit(3)
            ->get()
            ->map(function($destination) {
                return [
                    'id' => $destination->id,
                    'title' => $destination->title,
                    'location' => $destination->location,
                    'slug' => $destination->slug,
                    'type' => 'destination',
                    'display_text' => $destination->title . ' - ' . $destination->location,
                    'url' => route('destination.detail', $destination->slug)
                ];
            });

        // Search Articles
        $posts = Post::where('status', 'active')
            ->where(function($q) use ($query) {
                $q->where('title', 'like', '%' . $query . '%')
                  ->orWhere('summary', 'like', '%' . $query . '%');
            })
            ->select('id', 'title', 'slug')
            ->limit(3)
            ->get()
            ->map(function($post) {
                return [
                    'id' => $post->id,
                    'title' => $post->title,
                    'slug' => $post->slug,
                    'type' => 'article',
                    'display_text' => $post->title,
                    'url' => route('blog.detail', $post->slug)
                ];
            });

        $results = $tours->concat($destinations)->concat($posts)->take(8);
        
        return response()->json($results);
    }

}
