<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Document;
use Illuminate\Support\Str;

class DocumentController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $document=Document::orderBy('id','ASC')->paginate(57);
        return view('backend.document.index')->with('documents',$document);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.document.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // return $request->all();
        $this->validate($request,[
            'title'=>'string|required|max:100',
            'photo'=>'required|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'file' => 'required|mimes:pdf,doc,docx|max:2048',
            'status'=>'required|in:active,inactive',
        ]);
        $data=$request->all();
        $slug=Str::slug($request->title);
        $count=Document::where('slug',$slug)->count();
        if($count>0){
            $slug=$slug.'-'.date('ymdis').'-'.rand(0,999);
        }
        $data['slug']=$slug;
        $data['title'] = $request->get('title');
        $data['status'] = $request->get('status');
        // return $slug;
        $path = $request->file('photo')->store('public/images');
        $data['photo'] = $path; 

        $filePath = $request->file('file')->store('public/files');
        $data['file'] = $filePath;

        $status=Document::create($data);
        if($status){
            request()->session()->flash('success','Document successfully added');
        }
        else{
            request()->session()->flash('error','Error occurred while adding document');
        }
        return redirect()->route('document.index');

    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $document=Document::findOrFail($id);
        return view('backend.document.edit')
                    ->with('document',$document);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $document = Document::findOrFail($id);

        $this->validate($request, [
            'title' => 'string|required|max:100',
            'photo' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'file' => 'nullable|mimes:pdf,doc,docx|max:2048',
            'status' => 'required|in:active,inactive',
        ]);

        $data = $request->all();
        $data['title'] = $request->get('title');
        $data['status'] = $request->get('status');

        // Handle photo update
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('public/images');
            $data['photo'] = $path;
        } else {
            unset($data['photo']); // Remove the 'photo' key if it is not provided in the request
        }

        $filePath = $request->hasFile('file') ? $request->file('file')->store('public/files') : $document->file;
        $data['file'] = $filePath;

        $status = $document->fill($data)->save();

        if ($status) {
            request()->session()->flash('success', 'Document successfully updated');
        } else {
            request()->session()->flash('error', 'Error occurred while updating document');
        }

        return redirect()->route('document.index');
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $document=Document::findOrFail($id);
        $status=$document->delete();
        
        if($status){
            request()->session()->flash('success','Document successfully deleted');
        }
        else{
            request()->session()->flash('error','Error while deleting document');
        }
        return redirect()->route('document.index');
    }
}
