<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\Destination;
use App\Models\DestinationTag;
use App\User;

class DestinationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $destinations = Destination::orderBy('id', 'ASC')->paginate(57);
        // return $destinations;
        return view('backend.destination.index')->with('destinations',$destinations);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $tags=DestinationTag::get();
        $users=User::get();
        return view('backend.destination.create')->with('users',$users)->with('tags',$tags);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // return $request->all();
        $this->validate($request,[
            'title'=>'string|required',
            'summary'=>'string|nullable',
            'description'=>'string|nullable',
            'pros'=>'array|nullable',
            'pros.*'=>'string|nullable',
            'cons'=>'array|nullable',
            'cons.*'=>'string|nullable',
            'location'=>'string|required',
            'language'=>'string|required',
            'area'=>'string|required',
            'visa'=>'required|in:yes,no',
            'photo'=>'required|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'images.*' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'tags'=>'nullable',
            'added_by'=>'nullable',
            'status'=>'required|in:active,inactive,featured'
        ]);

        $data=$request->all();

        $slug=Str::slug($request->title);
        $count=Destination::where('slug',$slug)->count();
        if($count>0){
            $slug=$slug.'-'.date('ymdis').'-'.rand(0,999);
        }
        $data['slug']=$slug;
        $data['title'] = $request->get('title');
        $data['summary'] = $request->get('summary');
        $data['description'] = $request->get('description');
        
        // Handle pros and cons
        $pros = $request->input('pros', []);
        $cons = $request->input('cons', []);
        
        // Filter out empty values
        $data['pros'] = array_filter($pros, function($value) {
            return !empty(trim($value));
        });
        $data['cons'] = array_filter($cons, function($value) {
            return !empty(trim($value));
        });
        
        $data['location'] = $request->get('location');
        $data['language'] = $request->get('language');
        $data['age'] = $request->get('area');
        $data['visa'] = $request->get('visa');
        $data['added_by'] = $request->get('added_by');
        $data['status'] = $request->get('status');

        $tags=$request->input('tags',[]);
        if($tags){
            $data['tags']=implode(',',$tags);
        }
        else{
            $data['tags']='';
        }
        // return $data;
        $path = $request->file('photo')->store('public/images');
        $data['photo'] = $path;

        // Handle multiple image uploads
        $imagesPaths = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $imagesPaths[] = $image->store('public/multiple_images');
            }
        }
        $data['images'] = implode(',', $imagesPaths);

        $status=Destination::create($data);
        if($status){
            request()->session()->flash('success','Destination Successfully added');
        }
        else{
            request()->session()->flash('error','Please try again!!');
        }
        return redirect()->route('destination.index');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $destination=Destination::findOrFail($id);
        $tags=DestinationTag::get();
        $users=User::get();
        return view('backend.destination.edit')->with('users',$users)->with('tags',$tags)->with('destination',$destination);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $destination = Destination::findOrFail($id);

        $this->validate($request, [
            'title'=>'string|required',
            'summary'=>'string|nullable',
            'description'=>'string|nullable',
            'pros'=>'array|nullable',
            'pros.*'=>'string|nullable',
            'cons'=>'array|nullable',
            'cons.*'=>'string|nullable',
            'location'=>'string|required',
            'language'=>'string|required',
            'area'=>'string|required',
            'visa'=>'required|in:yes,no',
            'photo'=>'image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'images.*' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'tags'=>'nullable',
            'added_by'=>'nullable',
            'status'=>'required|in:active,inactive,featured'
        ]);

        $data = $request->all();
        $data['title'] = $request->get('title');
        $data['summary'] = $request->get('summary');
        $data['description'] = $request->get('description');
        
        // Handle pros and cons
        $pros = $request->input('pros', []);
        $cons = $request->input('cons', []);
        
        // Filter out empty values
        $data['pros'] = array_filter($pros, function($value) {
            return !empty(trim($value));
        });
        $data['cons'] = array_filter($cons, function($value) {
            return !empty(trim($value));
        });
        
        $data['location'] = $request->get('location');
        $data['language'] = $request->get('language');
        $data['area'] = $request->get('area');
        $data['visa'] = $request->get('visa');
        $data['added_by'] = $request->get('added_by');
        $data['status'] = $request->get('status');

        // Handle tags update
        $tags = $request->input('tags',[]);
        if ($tags) {
            $data['tags'] = implode(',', $tags);
        } else {
            $data['tags'] = '';
        }

        // Handle photo update only if a new one is provided
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('public/images');
            $data['photo'] = $path;
        } else {
            unset($data['photo']); // Remove the 'photo' key if it is not provided in the request
        }

        // Handle multiple image uploads
        $imagesPaths = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $imagesPaths[] = $image->store('public/multiple_images');
            }
            $data['images'] = implode(',', $imagesPaths);
        } else {
            unset($data['images']); // Remove the 'images' key if it is not provided in the request
        }

        $status = $destination->fill($data)->save();

        if ($status) {
            request()->session()->flash('success', 'Destination successfully updated');
        } else {
            request()->session()->flash('error', 'Please try again!!');
        }

        return redirect()->route('destination.index');
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $destination=Destination::findOrFail($id);
       
        $status=$destination->delete();
        
        if($status){
            request()->session()->flash('success','Destination successfully deleted');
        }
        else{
            request()->session()->flash('error','Error while deleting destination ');
        }
        return redirect()->route('destination.index');
    }
}
