<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use Illuminate\Support\Facades\Mail;
use App\Mail\BookingNotification;

class BookController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(){
        $orders=Order::paginate(20);
        return view('backend.order.index')->with('orders',$orders);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    public function store(Request $request)
    {
        $this->validate($request,[
            'first_name'=>'string|required|min:2',
            'last_name'=>'string|required|min:2',
            'email'=>'email|required',
            'phone'=>'numeric|required',
            'country'=>'string|required',
            'language' => 'string|required|max:100',
            'number_adult' => 'string|required|max:100',
            'number_child' => 'string|required|max:100',
            'age_child' => 'string|required|max:100',
            'date_arrival' => 'string|required|max:100',
            'total_stay' => 'string|required|max:100',
            'experience' => 'string|required|max:100',
            'accomodation' => 'string|required|max:100',
            'budget' => 'string|required|max:100',
            'like_most' => 'string|required|max:100',
            'dislike_most' => 'string|required|max:100',
            'brief_descript' => 'string|required|max:100',
            'tour_id' => 'required|exists:tours,id',
            'cf-turnstile-response'=>'required'
        ]);

        // Verify Turnstile token
        $turnstileResponse = $request->input('cf-turnstile-response');
        if (!$this->verifyTurnstile($turnstileResponse)) {
            request()->session()->flash('error','Security verification failed. Please try again.');
            return redirect()->route('book');
        }

        $order = Order::create($request->all());

        $data = array();
        $data['url'] = route('order.show', $order->id);
        $data['date'] = $order->created_at->format('F d, Y h:i A');
        $data['first_name'] = $order->first_name;
        $data['last_name'] = $order->last_name;
        $data['email'] = $order->email;
        $data['phone'] = $order->phone;
        $data['country'] = $order->country;
        $data['language'] = $order->language;
        $data['number_adult'] = $order->number_adult;
        $data['number_child'] = $order->number_child;
        $data['age_child'] = $order->age_child;
        $data['date_arrival'] = $order->date_arrival;
        $data['total_stay'] = $order->total_stay;
        $data['experience'] = $order->experience;
        $data['accomodation'] = $order->accomodation;
        $data['budget'] = $order->budget;
        $data['like_most'] = $order->like_most;
        $data['dislike_most'] = $order->dislike_most;
        $data['brief_descript'] = $order->brief_descript;
        $data['tour_id'] = $order->tour_id;

        // Mail::to('bookings@tembeahorizons.com')->send(new BookingNotification($order));

        if($order){
            request()->session()->flash('success','Booking Successfully sent');
            return view('frontend.pages.confirm-book',compact('order'));
        }
        else{
            request()->session()->flash('error','Please try again!!');
        }

        return redirect()->route('book');
    }

    /**
     * Verify Cloudflare Turnstile token
     *
     * @param string $token
     * @return bool
     */
    private function verifyTurnstile($token)
    {
        $secretKey = env('TURNSTILE_SECRET_KEY');
        
        if (!$secretKey) {
            // If no secret key is configured, skip verification in development
            return true;
        }

        $data = [
            'secret' => $secretKey,
            'response' => $token,
            'remoteip' => request()->ip()
        ];

        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data)
            ]
        ];

        $context = stream_context_create($options);
        $result = file_get_contents('https://challenges.cloudflare.com/turnstile/v0/siteverify', false, $context);
        
        if ($result === false) {
            return false;
        }

        $response = json_decode($result, true);
        return isset($response['success']) && $response['success'] === true;
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request,$id)
    {
        $order=Order::find($id);
        if($order){
            $order->created_at=\Carbon\Carbon::now();
            $order->save();
            return view('backend.order.show')->with('order',$order);
        }
        else{
            return back();
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $order=Order::find($id);
        $status=$order->delete();
        if($status){
            request()->session()->flash('success','Successfully deleted booking');
        }
        else{
            request()->session()->flash('error','Error occurred please try again');
        }
        return back();
    }
}
