<?php
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}
include '../includes/conn.php';
$isTeacher = isset($_SESSION['teachersys']);
$level = (int)($_SESSION['schoolsyslevel'] ?? 0);
$isAdminFull = isset($_SESSION['schoolsys']) && in_array($level, [1], true); // full admin (manage) - only super admin
$isPrincipal = isset($_SESSION['schoolsys']) && $level === 2; // principal - view only
$isHeadOfSection = isset($_SESSION['schoolsys']) && $level === 4; // view-only for discipline
$canViewPage = $isTeacher || $isAdminFull || $isPrincipal || $isHeadOfSection;
if (!$canViewPage) {
    header('Location: login.php');
    exit;
}

$teacher_id = $isTeacher ? (int)($_SESSION['teachersys'] ?? 0) : 0;
// threshold to consider repeat offender
$repeat_threshold = 3;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_case'])) {
    $student_id = isset($_POST['student_id']) ? (int)$_POST['student_id'] : 0;
    $incident_type = trim((string)($_POST['incident_type'] ?? ''));
    $description = trim((string)($_POST['description'] ?? ''));
    $action_taken = trim((string)($_POST['action_taken'] ?? ''));
    $severity = isset($_POST['severity']) ? (int)$_POST['severity'] : 1;

    $errors = [];
    if ($student_id <= 0) $errors[] = 'Select a valid student.';
    if ($incident_type === '') $errors[] = 'Enter incident type.';
    if ($description === '') $errors[] = 'Enter a description.';

    if (empty($errors)) {
        $stmt = mysqli_prepare($con, "INSERT INTO discipline_cases(student_id,reporter_id,`date`,incident_type,description,action_taken,severity,status) VALUES(?,?,?,?,?,?,?,1)");
        if ($stmt) {
            $now = date('Y-m-d H:i:s');
            mysqli_stmt_bind_param($stmt, 'iissssi', $student_id, $teacher_id, $now, $incident_type, $description, $action_taken, $severity);
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
            $success_msg = 'Discipline case recorded.';
        } else {
            $errors[] = 'Unable to save case.';
        }
    }
}

// Pagination / listing: get latest 50 active cases (status=1)
$cases = [];
$cases_q = mysqli_query($con, "SELECT dc.*, s.fullname, s.regnumber FROM discipline_cases dc LEFT JOIN students s ON s.student_id = dc.student_id WHERE dc.status=1 ORDER BY dc.date DESC LIMIT 50");
if ($cases_q) {
    while ($r = mysqli_fetch_assoc($cases_q)) $cases[] = $r;
}

// Get counts per student to highlight repeat offenders (only active cases)
$counts = [];
$cnt_q = mysqli_query($con, "SELECT student_id, COUNT(*) AS c FROM discipline_cases WHERE status=1 GROUP BY student_id");
if ($cnt_q) {
    while ($r = mysqli_fetch_assoc($cnt_q)) {
        $counts[(int)$r['student_id']] = (int)$r['c'];
    }
}

// Simple student list for form select (class students only if teacher is classteacher)
$students = [];
if (isset($_SESSION['class_id']) && (int)$_SESSION['class_id'] > 0) {
    $cid = (int)$_SESSION['class_id'];
    $s_q = mysqli_query($con, "SELECT student_id,fullname,regnumber FROM students WHERE class_id='{$cid}' AND status=1 ORDER BY fullname");
} else {
    $s_q = mysqli_query($con, "SELECT student_id,fullname,regnumber FROM students WHERE status=1 ORDER BY fullname LIMIT 200");
}
if ($s_q) {
    while ($r = mysqli_fetch_assoc($s_q)) $students[] = $r;
}

?>
<!DOCTYPE html>
<html>
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Discipline Cases - School Manager</title>
<link href="<?php echo BASE_URL; ?>/css/style.css" rel="stylesheet">
<link href="<?php echo BASE_URL; ?>/css/style-responsive.css" rel="stylesheet">
</head>
<body class="sticky-header">
<section>
<?php
    // if admin-level user (including Principal and Head of Section) is viewing, include the admin header (shows admin sidebar);
    // otherwise use teacher nav. Principal and Head of Section should see the admin sidebar but will have view-only access on this page.
    if (!empty($isAdminFull) || !empty($isPrincipal) || !empty($isHeadOfSection)) {
        include __DIR__ . '/../includes/header.php';
    } else {
        include 'nav.php';
    }
?>

<section class="wrapper">
    <div class="row">
        <div class="col-lg-4">
            <section class="panel">
                <header class="panel-heading">Add Discipline Case</header>
                <div class="panel-body">
                    <?php
                    if (!empty($errors)) {
                        foreach ($errors as $e) echo '<div class="alert alert-danger">' . htmlspecialchars($e) . '</div>';
                    }
                    if (!empty($success_msg)) echo '<div class="alert alert-success">' . htmlspecialchars($success_msg) . '</div>';
                    ?>
                    <?php // Only teachers and full admins may add cases. Head of Section can only view. ?>
                    <?php if ($isTeacher || $isAdminFull): ?>
                    <form method="POST" class="form">
                        <div class="form-group">
                            <label>Student</label>
                            <select name="student_id" class="form-control" required>
                                <option value="">Select student</option>
                                <?php foreach ($students as $st): ?>
                                    <option value="<?php echo (int)$st['student_id']; ?>"><?php echo htmlspecialchars($st['fullname'] . ' (' . $st['regnumber'] . ')'); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Incident Type</label>
                            <input type="text" name="incident_type" class="form-control" required>
                        </div>
                        <div class="form-group">
                            <label>Description</label>
                            <textarea name="description" class="form-control" rows="4" required></textarea>
                        </div>
                        <div class="form-group">
                            <label>Action Taken</label>
                            <input type="text" name="action_taken" class="form-control">
                        </div>
                        <div class="form-group">
                            <label>Severity (1-5)</label>
                            <input type="number" name="severity" class="form-control" min="1" max="5" value="1">
                        </div>
                        <div class="form-group">
                            <button class="btn btn-primary" name="add_case" type="submit">Submit</button>
                        </div>
                    </form>
                    <?php else: ?>
                        <div class="alert alert-info">You have view-only access to discipline cases.</div>
                    <?php endif; ?>
                </div>
            </section>
        </div>
        <div class="col-lg-8">
            <section class="panel">
                <header class="panel-heading">Recent Discipline Cases</header>
                <div class="panel-body">
                    <table class="table table-striped table-bordered">
                        <thead>
                            <tr>
                                <th>Student</th>
                                <th>Date</th>
                                <th>Type</th>
                                <th>Severity</th>
                                <th>Reporter</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($cases)): ?>
                                <tr><td colspan="5">No cases recorded.</td></tr>
                            <?php else: ?>
                                <?php foreach ($cases as $c):
                                    $sid = (int)$c['student_id'];
                                    $count = isset($counts[$sid]) ? $counts[$sid] : 0;
                                    $repeat = $count >= $repeat_threshold;
                                ?>
                                    <tr style="<?php echo $repeat ? 'background:#f8d7da' : ''; ?>">
                                        <td><a href="discipline_details?id=<?php echo (int)$c['id']; ?>"><?php echo htmlspecialchars($c['fullname'] . (trim((string)($c['regnumber'] ?? '')) !== '' ? ' (' . $c['regnumber'] . ')' : '')); ?></a>
                                            <?php if ($repeat) echo ' <span class="badge" style="background:#721c24;color:#fff;">Repeat (' . $count . ')</span>'; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($c['date']); ?></td>
                                        <td><?php echo htmlspecialchars($c['incident_type']); ?></td>
                                        <td><?php echo (int)$c['severity']; ?></td>
                                        <td><?php
                                            // load reporter name
                                            $rep = '';
                                            if (!empty($c['reporter_id'])) {
                                                $rq = mysqli_query($con, "SELECT fullname FROM teachers WHERE teacher_id='" . (int)$c['reporter_id'] . "' LIMIT 1");
                                                if ($rq && $rr = mysqli_fetch_assoc($rq)) $rep = $rr['fullname'] ?? '';
                                            }
                                            echo htmlspecialchars($rep);
                                        ?></td>
                                        <td>
                                            <?php
                                            // manage = edit/delete allowed only for full admins (level 1) or the reporting teacher
                                            $canManage = false;
                                            if ($isAdminFull) $canManage = true;
                                            if ($isTeacher && !empty($c['reporter_id']) && (int)$c['reporter_id'] === $teacher_id) $canManage = true;
                                            // Head of Section and Principal have view-only rights: show a View link but no edit/delete
                                            if ($canManage): ?>
                                                <a href="discipline_edit?id=<?php echo (int)$c['id']; ?>" class="btn btn-xs btn-primary">Edit</a>
                                                <a href="discipline_delete?id=<?php echo (int)$c['id']; ?>" class="btn btn-xs btn-danger" onclick="return confirm('Are you sure you want to delete this case?');">Delete</a>
                                            <?php elseif (!empty($isHeadOfSection) || !empty($isPrincipal)): ?>
                                                <a href="discipline_details?id=<?php echo (int)$c['id']; ?>" class="btn btn-xs btn-default">View</a>
                                            <?php else: ?>
                                                &ndash;
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </section>
        </div>
    </div>
</section>
</section>

<script src="<?php echo BASE_URL; ?>/js/jquery-1.11.3.min.js"></script>
<script src="<?php echo BASE_URL; ?>/js/bootstrap.min.js"></script>
</body>
</html>