<?php
// Migration script to convert weeklyplans startdate and enddate from UNIX timestamps to DATE format
require_once dirname(__DIR__) . '/includes/conn.php';

echo "=== Weekly Plans Date Migration ===\n\n";

// Check current state
echo "Checking current data...\n";
$check = mysqli_query($con, "SELECT weeklyplan_id, startdate, enddate FROM weeklyplans LIMIT 5");
if ($check) {
    echo "Sample data:\n";
    while ($row = mysqli_fetch_assoc($check)) {
        echo "  ID: {$row['weeklyplan_id']}, Start: {$row['startdate']}, End: {$row['enddate']}\n";
    }
    echo "\n";
}

// Preview conversion
echo "Previewing conversion (numeric timestamps only)...\n";
$preview = mysqli_query($con, "
    SELECT weeklyplan_id, startdate, FROM_UNIXTIME(startdate) AS start_human, 
           enddate, FROM_UNIXTIME(enddate) AS end_human
    FROM weeklyplans
    WHERE (startdate REGEXP '^[0-9]{9,10}$' OR enddate REGEXP '^[0-9]{9,10}$')
    LIMIT 10
");
if ($preview) {
    $count = 0;
    while ($row = mysqli_fetch_assoc($preview)) {
        echo "  ID {$row['weeklyplan_id']}: {$row['startdate']} -> {$row['start_human']}, {$row['enddate']} -> {$row['end_human']}\n";
        $count++;
    }
    echo "Found $count rows with numeric timestamps\n\n";
}

// Ask for confirmation
echo "Proceed with conversion? (yes/no): ";
$handle = fopen("php://stdin", "r");
$line = trim(fgets($handle));
fclose($handle);

if (strtolower($line) !== 'yes') {
    echo "Migration cancelled.\n";
    exit(0);
}

// Start transaction
mysqli_begin_transaction($con);

try {
    echo "\nConverting startdate...\n";
    $result1 = mysqli_query($con, "
        UPDATE weeklyplans
        SET startdate = FROM_UNIXTIME(startdate, '%Y-%m-%d')
        WHERE startdate REGEXP '^[0-9]{9,10}$'
    ");
    if ($result1) {
        $affected1 = mysqli_affected_rows($con);
        echo "  Converted $affected1 startdate values\n";
    } else {
        throw new Exception("Error converting startdate: " . mysqli_error($con));
    }

    echo "Converting enddate...\n";
    $result2 = mysqli_query($con, "
        UPDATE weeklyplans
        SET enddate = FROM_UNIXTIME(enddate, '%Y-%m-%d')
        WHERE enddate REGEXP '^[0-9]{9,10}$'
    ");
    if ($result2) {
        $affected2 = mysqli_affected_rows($con);
        echo "  Converted $affected2 enddate values\n";
    } else {
        throw new Exception("Error converting enddate: " . mysqli_error($con));
    }

    // Commit transaction
    mysqli_commit($con);
    echo "\nMigration completed successfully!\n\n";

    // Verify results
    echo "Verification:\n";
    $verify = mysqli_query($con, "
        SELECT COUNT(*) AS total_rows, 
        SUM(startdate REGEXP '^[0-9]{4}-[0-9]{2}-[0-9]{2}$') AS startdate_converted,
        SUM(enddate REGEXP '^[0-9]{4}-[0-9]{2}-[0-9]{2}$') AS enddate_converted
        FROM weeklyplans
    ");
    if ($verify) {
        $stats = mysqli_fetch_assoc($verify);
        echo "  Total rows: {$stats['total_rows']}\n";
        echo "  Startdate in DATE format: {$stats['startdate_converted']}\n";
        echo "  Enddate in DATE format: {$stats['enddate_converted']}\n";
    }

} catch (Exception $e) {
    mysqli_rollback($con);
    echo "\nMigration failed: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\nDone!\n";
?>
