-- ============================================================================
-- FAMILY DISCOUNT & PAYMENT PENALTIES ENHANCEMENT
-- Additional tables and updates for comprehensive discount system
-- ============================================================================

-- Add family_id column to students table if not exists
ALTER TABLE students ADD COLUMN IF NOT EXISTS family_id INT(11) DEFAULT 0 AFTER stream_id;
ALTER TABLE students ADD INDEX IF NOT EXISTS idx_family_id (family_id);

-- ============================================================================
-- FAMILIES TABLE - Track family groupings
-- ============================================================================
CREATE TABLE IF NOT EXISTS `families` (
  `family_id` INT(11) NOT NULL AUTO_INCREMENT,
  `family_name` VARCHAR(200),
  `primary_contact_name` VARCHAR(150),
  `primary_contact_phone` VARCHAR(20),
  `primary_contact_email` VARCHAR(100),
  `secondary_contact_phone` VARCHAR(20),
  `secondary_contact_email` VARCHAR(100),
  `address` TEXT,
  `total_students` INT(11) DEFAULT 0,
  `discount_eligible_count` INT(11) DEFAULT 0 COMMENT 'Students excluding nursery',
  `notes` TEXT,
  `status` INT(2) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`family_id`),
  INDEX `idx_status` (`status`),
  INDEX `idx_phone` (`primary_contact_phone`),
  INDEX `idx_email` (`primary_contact_email`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- PAYMENT ADJUSTMENTS TABLE - Track early payments and late penalties
-- ============================================================================
CREATE TABLE IF NOT EXISTS `payment_adjustments` (
  `adjustment_id` INT(11) NOT NULL AUTO_INCREMENT,
  `student_id` INT(11) NOT NULL,
  `term_id` INT(11) NOT NULL,
  `adjustment_type` ENUM('Early Payment Discount', 'Late Payment Penalty', 'Sibling Discount', 'Scholarship', 'Waiver', 'Other') NOT NULL,
  `percentage` DECIMAL(5, 2) DEFAULT 0 COMMENT 'Percentage of adjustment',
  `amount` DECIMAL(15, 2) DEFAULT 0 COMMENT 'Fixed amount adjustment',
  `base_fee_amount` DECIMAL(15, 2) NOT NULL COMMENT 'Original fee before adjustment',
  `adjusted_amount` DECIMAL(15, 2) NOT NULL COMMENT 'Amount after adjustment',
  `reason` TEXT,
  `payment_date` DATE COMMENT 'Date of payment (for early/late calculation)',
  `term_start_date` DATE COMMENT 'Term start date (for early payment check)',
  `half_term_date` DATE COMMENT 'Half term date (for late penalty check)',
  `is_new_student` INT(2) DEFAULT 0 COMMENT '1 if new student (excludes early discount)',
  `applied_by` INT(11) COMMENT 'User ID who applied adjustment',
  `status` INT(2) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`adjustment_id`),
  KEY `fk_student_adj` (`student_id`),
  KEY `fk_term_adj` (`term_id`),
  INDEX `idx_adjustment_type` (`adjustment_type`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- STUDENT FEE CALCULATIONS TABLE - Pre-calculated fee details per student/term
-- ============================================================================
CREATE TABLE IF NOT EXISTS `student_fee_calculations` (
  `calculation_id` INT(11) NOT NULL AUTO_INCREMENT,
  `student_id` INT(11) NOT NULL,
  `term_id` INT(11) NOT NULL,
  `class_id` INT(11) NOT NULL,
  `base_fee_amount` DECIMAL(15, 2) NOT NULL COMMENT 'Original school fee',
  `sibling_discount_percentage` DECIMAL(5, 2) DEFAULT 0,
  `sibling_discount_amount` DECIMAL(15, 2) DEFAULT 0,
  `early_payment_discount` DECIMAL(15, 2) DEFAULT 0,
  `late_payment_penalty` DECIMAL(15, 2) DEFAULT 0,
  `other_discounts` DECIMAL(15, 2) DEFAULT 0,
  `total_discounts` DECIMAL(15, 2) DEFAULT 0,
  `total_penalties` DECIMAL(15, 2) DEFAULT 0,
  `final_fee_amount` DECIMAL(15, 2) NOT NULL COMMENT 'Final amount to be paid',
  `total_paid` DECIMAL(15, 2) DEFAULT 0,
  `balance` DECIMAL(15, 2) DEFAULT 0,
  `overpayment` DECIMAL(15, 2) DEFAULT 0 COMMENT 'Amount paid in excess',
  `payment_status` ENUM('Unpaid', 'Partial', 'Paid', 'Overpaid', 'Overdue') DEFAULT 'Unpaid',
  `is_nursery_student` INT(2) DEFAULT 0 COMMENT '1 if student in nursery class',
  `is_new_student` INT(2) DEFAULT 0 COMMENT '1 if student joined this term',
  `family_id` INT(11) DEFAULT 0,
  `sibling_position` INT(11) DEFAULT 0 COMMENT 'Position in family (1st, 2nd, 3rd child)',
  `last_calculated` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  `status` INT(2) DEFAULT 1,
  PRIMARY KEY (`calculation_id`),
  UNIQUE KEY `unique_student_term` (`student_id`, `term_id`),
  KEY `fk_student_calc` (`student_id`),
  KEY `fk_term_calc` (`term_id`),
  KEY `fk_class_calc` (`class_id`),
  KEY `fk_family_calc` (`family_id`),
  INDEX `idx_payment_status` (`payment_status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- PAYMENT BANK DETAILS TABLE - Link payments to bank transactions
-- ============================================================================
CREATE TABLE IF NOT EXISTS `payment_bank_details` (
  `payment_bank_id` INT(11) NOT NULL AUTO_INCREMENT,
  `feespayment_id` INT(11) NOT NULL,
  `student_id` INT(11) NOT NULL,
  `bank_account_id` INT(11) COMMENT 'Which school bank account received payment',
  `bank_transaction_id` INT(11) COMMENT 'Link to bank_transactions table',
  `payment_method_id` INT(11) COMMENT 'Link to payment_methods table',
  `transaction_reference` VARCHAR(100) COMMENT 'Bank reference number',
  `payer_name` VARCHAR(150) COMMENT 'Name of person who paid',
  `payer_phone` VARCHAR(20),
  `payment_date` DATE NOT NULL,
  `amount` DECIMAL(15, 2) NOT NULL,
  `reconciled` INT(2) DEFAULT 0 COMMENT '1 if reconciled with bank statement',
  `reconciled_by` INT(11),
  `reconciled_date` TIMESTAMP NULL,
  `notes` TEXT,
  `status` INT(2) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`payment_bank_id`),
  KEY `fk_feespayment` (`feespayment_id`),
  KEY `fk_student_bank` (`student_id`),
  KEY `fk_bank_account` (`bank_account_id`),
  KEY `fk_bank_transaction` (`bank_transaction_id`),
  INDEX `idx_reconciled` (`reconciled`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TERM DATES TABLE - Store important term dates for penalty calculations
-- ============================================================================
CREATE TABLE IF NOT EXISTS `term_important_dates` (
  `term_date_id` INT(11) NOT NULL AUTO_INCREMENT,
  `term_id` INT(11) NOT NULL,
  `term_start_date` DATE NOT NULL,
  `term_end_date` DATE NOT NULL,
  `half_term_date` DATE COMMENT 'Half term date (for late penalty)',
  `late_penalty_start_date` DATE COMMENT '1 week after half term',
  `early_payment_deadline` DATE COMMENT 'Date before which early discount applies',
  `academic_year_id` INT(11),
  `notes` TEXT,
  `status` INT(2) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`term_date_id`),
  UNIQUE KEY `unique_term` (`term_id`),
  KEY `fk_term_dates` (`term_id`),
  INDEX `idx_term_start` (`term_start_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- Insert default payment methods if not exists
-- ============================================================================
INSERT IGNORE INTO payment_methods (method_name, method_code, description, status) VALUES
('Bank Transfer', 'BANK_TRANSFER', 'Direct bank transfer or deposit', 1),
('Check', 'CHECK', 'Check payment', 1),
('Cash', 'CASH', 'Cash payment at school office', 1),
('Mobile Money', 'MOBILE_MONEY', 'M-Pesa, Airtel Money, etc.', 1),
('Credit/Debit Card', 'CARD', 'Card payment', 1),
('Direct Debit', 'DIRECT_DEBIT', 'Automatic bank deduction', 1);

-- ============================================================================
-- Create view for easy family discount overview
-- ============================================================================
CREATE OR REPLACE VIEW `v_family_discount_summary` AS
SELECT 
    f.family_id,
    f.family_name,
    f.primary_contact_name,
    f.primary_contact_phone,
    f.total_students,
    f.discount_eligible_count,
    COUNT(DISTINCT s.student_id) AS actual_student_count,
    COUNT(DISTINCT CASE WHEN c.class NOT LIKE '%nursery%' THEN s.student_id END) AS non_nursery_count,
    GROUP_CONCAT(DISTINCT s.fullname ORDER BY s.student_id SEPARATOR ', ') AS student_names,
    GROUP_CONCAT(DISTINCT c.class ORDER BY s.student_id SEPARATOR ', ') AS student_classes
FROM families f
LEFT JOIN students s ON s.family_id = f.family_id AND s.status = 1
LEFT JOIN classes c ON s.class_id = c.class_id
WHERE f.status = 1
GROUP BY f.family_id;

-- ============================================================================
-- Create view for student payment details with all adjustments
-- ============================================================================
CREATE OR REPLACE VIEW `v_student_payment_details` AS
SELECT 
    sfc.calculation_id,
    sfc.student_id,
    s.fullname,
    s.regnumber,
    sfc.term_id,
    CONCAT('Term ', ft.term, ' ', ft.year) AS term_name,
    sfc.class_id,
    c.class AS class_name,
    sfc.base_fee_amount,
    sfc.sibling_discount_percentage,
    sfc.sibling_discount_amount,
    sfc.early_payment_discount,
    sfc.late_payment_penalty,
    sfc.other_discounts,
    sfc.total_discounts,
    sfc.total_penalties,
    sfc.final_fee_amount,
    sfc.total_paid,
    sfc.balance,
    sfc.overpayment,
    sfc.payment_status,
    sfc.family_id,
    f.family_name,
    sfc.sibling_position,
    CASE 
        WHEN sfc.sibling_position = 1 THEN '1st Child (No Discount)'
        WHEN sfc.sibling_position = 2 THEN '2nd Child (10% Discount)'
        WHEN sfc.sibling_position >= 3 THEN CONCAT(sfc.sibling_position, 'rd+ Child (20% Discount)')
        ELSE 'Not in Family'
    END AS sibling_discount_description,
    sfc.last_calculated
FROM student_fee_calculations sfc
JOIN students s ON sfc.student_id = s.student_id
JOIN financialterms ft ON sfc.term_id = ft.financialterm_id
JOIN classes c ON sfc.class_id = c.class_id
LEFT JOIN families f ON sfc.family_id = f.family_id
WHERE sfc.status = 1 AND s.status = 1;

-- ============================================================================
-- Insert sample term dates for current/upcoming terms
-- ============================================================================
-- Note: Adjust dates according to your school calendar
INSERT IGNORE INTO term_important_dates (term_id, term_start_date, term_end_date, half_term_date, late_penalty_start_date, early_payment_deadline, status)
SELECT 
    financialterm_id,
    DATE(FROM_UNIXTIME(startdate)) AS term_start_date,
    DATE(FROM_UNIXTIME(enddate)) AS term_end_date,
    DATE_ADD(DATE(FROM_UNIXTIME(startdate)), INTERVAL 6 WEEK) AS half_term_date,
    DATE_ADD(DATE_ADD(DATE(FROM_UNIXTIME(startdate)), INTERVAL 6 WEEK), INTERVAL 7 DAY) AS late_penalty_start_date,
    DATE_SUB(DATE(FROM_UNIXTIME(startdate)), INTERVAL 1 DAY) AS early_payment_deadline,
    1
FROM financialterms
WHERE status = 1;

-- ============================================================================
-- STORED PROCEDURE: Calculate student fees with all discounts/penalties
-- ============================================================================
DELIMITER $$

CREATE PROCEDURE IF NOT EXISTS `sp_calculate_student_fees`(
    IN p_student_id INT,
    IN p_term_id INT
)
BEGIN
    DECLARE v_base_fee DECIMAL(15,2) DEFAULT 0;
    DECLARE v_class_id INT DEFAULT 0;
    DECLARE v_family_id INT DEFAULT 0;
    DECLARE v_is_nursery INT DEFAULT 0;
    DECLARE v_is_new_student INT DEFAULT 0;
    DECLARE v_sibling_position INT DEFAULT 0;
    DECLARE v_sibling_discount_pct DECIMAL(5,2) DEFAULT 0;
    DECLARE v_sibling_discount_amt DECIMAL(15,2) DEFAULT 0;
    DECLARE v_total_paid DECIMAL(15,2) DEFAULT 0;
    DECLARE v_final_fee DECIMAL(15,2) DEFAULT 0;
    DECLARE v_balance DECIMAL(15,2) DEFAULT 0;
    DECLARE v_overpayment DECIMAL(15,2) DEFAULT 0;
    DECLARE v_payment_status VARCHAR(20) DEFAULT 'Unpaid';
    
    -- Get student and class information
    SELECT s.class_id, s.family_id, s.joiningterm,
           CASE WHEN LOWER(c.class) LIKE '%nursery%' THEN 1 ELSE 0 END,
           CASE WHEN s.joiningterm = p_term_id THEN 1 ELSE 0 END
    INTO v_class_id, v_family_id, v_is_new_student, v_is_nursery, v_is_new_student
    FROM students s
    JOIN classes c ON s.class_id = c.class_id
    WHERE s.student_id = p_student_id AND s.status = 1;
    
    -- Get base fee for class/year
    SELECT sf.amount INTO v_base_fee
    FROM schoolfees sf
    JOIN financialterms ft ON ft.year = sf.year
    WHERE sf.class_id = v_class_id 
    AND ft.financialterm_id = p_term_id
    LIMIT 1;
    
    IF v_base_fee IS NULL THEN
        SET v_base_fee = 0;
    END IF;
    
    -- Calculate sibling discount (only if not nursery and has family)
    IF v_family_id > 0 AND v_is_nursery = 0 THEN
        -- Get sibling position (count non-nursery siblings)
        SELECT COUNT(*) + 1 INTO v_sibling_position
        FROM students s2
        JOIN classes c2 ON s2.class_id = c2.class_id
        WHERE s2.family_id = v_family_id
        AND s2.student_id < p_student_id
        AND s2.status = 1
        AND LOWER(c2.class) NOT LIKE '%nursery%';
        
        -- Apply discount based on position
        IF v_sibling_position = 2 THEN
            SET v_sibling_discount_pct = 10.00;
        ELSEIF v_sibling_position >= 3 THEN
            SET v_sibling_discount_pct = 20.00;
        END IF;
        
        SET v_sibling_discount_amt = (v_base_fee * v_sibling_discount_pct) / 100;
    END IF;
    
    -- Calculate final fee
    SET v_final_fee = v_base_fee - v_sibling_discount_amt;
    
    -- Get total paid
    SELECT COALESCE(SUM(amount), 0) INTO v_total_paid
    FROM feespayments
    WHERE student_id = p_student_id 
    AND term_id = p_term_id 
    AND status = 1;
    
    -- Calculate balance/overpayment
    IF v_total_paid > v_final_fee THEN
        SET v_overpayment = v_total_paid - v_final_fee;
        SET v_balance = 0;
        SET v_payment_status = 'Overpaid';
    ELSEIF v_total_paid = v_final_fee THEN
        SET v_balance = 0;
        SET v_payment_status = 'Paid';
    ELSEIF v_total_paid > 0 THEN
        SET v_balance = v_final_fee - v_total_paid;
        SET v_payment_status = 'Partial';
    ELSE
        SET v_balance = v_final_fee;
        SET v_payment_status = 'Unpaid';
    END IF;
    
    -- Insert or update calculation record
    INSERT INTO student_fee_calculations (
        student_id, term_id, class_id, base_fee_amount,
        sibling_discount_percentage, sibling_discount_amount,
        total_discounts, final_fee_amount, total_paid, balance, overpayment,
        payment_status, is_nursery_student, is_new_student, family_id, sibling_position
    ) VALUES (
        p_student_id, p_term_id, v_class_id, v_base_fee,
        v_sibling_discount_pct, v_sibling_discount_amt,
        v_sibling_discount_amt, v_final_fee, v_total_paid, v_balance, v_overpayment,
        v_payment_status, v_is_nursery, v_is_new_student, v_family_id, v_sibling_position
    )
    ON DUPLICATE KEY UPDATE
        class_id = v_class_id,
        base_fee_amount = v_base_fee,
        sibling_discount_percentage = v_sibling_discount_pct,
        sibling_discount_amount = v_sibling_discount_amt,
        total_discounts = v_sibling_discount_amt,
        final_fee_amount = v_final_fee,
        total_paid = v_total_paid,
        balance = v_balance,
        overpayment = v_overpayment,
        payment_status = v_payment_status,
        family_id = v_family_id,
        sibling_position = v_sibling_position;
END$$

DELIMITER ;

-- ============================================================================
-- COMMENTS
-- ============================================================================
-- This enhanced schema provides:
-- 1. Family tracking with automatic sibling discount (10% 2nd, 20% 3rd+)
-- 2. Payment adjustments for early/late payments
-- 3. Comprehensive fee calculations per student/term
-- 4. Bank payment tracking and reconciliation
-- 5. Term dates for penalty calculations
-- 6. Views for easy reporting
-- 7. Stored procedure for automated fee calculation
--
-- Usage:
-- - Family grouping happens automatically in addstudent.php
-- - Call sp_calculate_student_fees(student_id, term_id) to update calculations
-- - Query v_student_payment_details for comprehensive fee information
-- ============================================================================
