-- SQL Schema for PAYE Tax Brackets, Transportation, and Facilities Management
-- Run this before using the new modules

-- 1. PAYE TAX BRACKETS TABLE
-- Store Kenya Revenue Authority (KRA) tax rates
CREATE TABLE IF NOT EXISTS `paye_tax_brackets` (
  `bracket_id` INT(11) NOT NULL AUTO_INCREMENT,
  `bracket_name` VARCHAR(100) NOT NULL COMMENT 'e.g., Band 1, Band 2',
  `min_amount` DECIMAL(15,2) NOT NULL DEFAULT 0 COMMENT 'Minimum taxable income',
  `max_amount` DECIMAL(15,2) NULL COMMENT 'Maximum taxable income (NULL = unlimited)',
  `rate` DECIMAL(5,2) NOT NULL COMMENT 'Tax percentage (e.g., 10.00 for 10%)',
  `effective_date` DATE NOT NULL COMMENT 'When this rate became effective',
  `status` TINYINT(1) NOT NULL DEFAULT 1 COMMENT '1=Active, 0=Inactive',
  `notes` TEXT NULL,
  `created_by` INT(11) NULL,
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`bracket_id`),
  KEY `idx_status` (`status`),
  KEY `idx_effective_date` (`effective_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default Kenya 2025 PAYE tax brackets
INSERT INTO `paye_tax_brackets` (`bracket_name`, `min_amount`, `max_amount`, `rate`, `effective_date`, `notes`) VALUES
('Band 1: Up to KES 24,000', 0.00, 24000.00, 10.00, '2025-01-01', 'First KES 24,000 per month'),
('Band 2: KES 24,001 - 32,333', 24000.01, 32333.00, 25.00, '2025-01-01', 'Next KES 8,333 per month'),
('Band 3: KES 32,334 - 500,000', 32333.01, 500000.00, 30.00, '2025-01-01', 'Next KES 467,667 per month'),
('Band 4: KES 500,001 - 800,000', 500000.01, 800000.00, 32.50, '2025-01-01', 'Next KES 300,000 per month'),
('Band 5: Above KES 800,000', 800000.01, NULL, 35.00, '2025-01-01', 'All income above KES 800,000 per month');

-- 2. VEHICLES TABLE (Transportation Management)
CREATE TABLE IF NOT EXISTS `vehicles` (
  `vehicle_id` INT(11) NOT NULL AUTO_INCREMENT,
  `vehicle_number` VARCHAR(50) NOT NULL UNIQUE COMMENT 'Registration/plate number',
  `vehicle_type` ENUM('Bus','Van','Car','Truck','Motorcycle','Other') NOT NULL DEFAULT 'Bus',
  `make` VARCHAR(100) NULL COMMENT 'Manufacturer',
  `model` VARCHAR(100) NULL,
  `year` INT(4) NULL,
  `capacity` INT(11) NULL COMMENT 'Passenger capacity',
  `fuel_type` ENUM('Petrol','Diesel','Electric','Hybrid','Other') NULL,
  `vin_number` VARCHAR(50) NULL COMMENT 'Vehicle Identification Number',
  `insurance_company` VARCHAR(200) NULL,
  `insurance_policy` VARCHAR(100) NULL,
  `insurance_expiry` DATE NULL,
  `license_expiry` DATE NULL,
  `purchase_date` DATE NULL,
  `purchase_cost` DECIMAL(15,2) NULL,
  `current_mileage` INT(11) NULL DEFAULT 0,
  `last_service_date` DATE NULL,
  `next_service_date` DATE NULL,
  `assigned_driver` VARCHAR(200) NULL,
  `status` ENUM('Active','In Maintenance','Retired','Out of Service') NOT NULL DEFAULT 'Active',
  `notes` TEXT NULL,
  `created_by` INT(11) NULL,
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`vehicle_id`),
  KEY `idx_status` (`status`),
  KEY `idx_vehicle_type` (`vehicle_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 3. VEHICLE MAINTENANCE TABLE
CREATE TABLE IF NOT EXISTS `vehicle_maintenance` (
  `maintenance_id` INT(11) NOT NULL AUTO_INCREMENT,
  `vehicle_id` INT(11) NOT NULL,
  `maintenance_type` ENUM('Scheduled Service','Repair','Inspection','Fuel','Tire Change','Other') NOT NULL,
  `maintenance_date` DATE NOT NULL,
  `description` TEXT NOT NULL,
  `cost` DECIMAL(15,2) NOT NULL DEFAULT 0,
  `mileage_at_service` INT(11) NULL,
  `service_provider` VARCHAR(200) NULL,
  `invoice_number` VARCHAR(100) NULL,
  `next_service_due` DATE NULL,
  `performed_by` VARCHAR(200) NULL,
  `status` ENUM('Scheduled','Completed','Cancelled') NOT NULL DEFAULT 'Completed',
  `notes` TEXT NULL,
  `created_by` INT(11) NULL,
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`maintenance_id`),
  KEY `fk_vehicle_maintenance` (`vehicle_id`),
  KEY `idx_maintenance_date` (`maintenance_date`),
  CONSTRAINT `fk_vehicle_maintenance` FOREIGN KEY (`vehicle_id`) REFERENCES `vehicles` (`vehicle_id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 4. BUS ROUTES TABLE
CREATE TABLE IF NOT EXISTS `bus_routes` (
  `route_id` INT(11) NOT NULL AUTO_INCREMENT,
  `route_name` VARCHAR(200) NOT NULL,
  `vehicle_id` INT(11) NULL,
  `route_description` TEXT NULL,
  `pickup_points` TEXT NULL COMMENT 'JSON or comma-separated list',
  `estimated_time` INT(11) NULL COMMENT 'Minutes',
  `morning_departure` TIME NULL,
  `afternoon_departure` TIME NULL,
  `monthly_fee` DECIMAL(10,2) NULL,
  `status` TINYINT(1) NOT NULL DEFAULT 1 COMMENT '1=Active, 0=Inactive',
  `created_by` INT(11) NULL,
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`route_id`),
  KEY `fk_route_vehicle` (`vehicle_id`),
  CONSTRAINT `fk_route_vehicle` FOREIGN KEY (`vehicle_id`) REFERENCES `vehicles` (`vehicle_id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 5. FACILITIES TABLE
CREATE TABLE IF NOT EXISTS `facilities` (
  `facility_id` INT(11) NOT NULL AUTO_INCREMENT,
  `facility_name` VARCHAR(200) NOT NULL,
  `facility_type` ENUM('Building','Classroom','Laboratory','Library','Office','Sports Field','Playground','Cafeteria','Dormitory','Other') NOT NULL,
  `location` VARCHAR(200) NULL COMMENT 'Building/floor/room number',
  `capacity` INT(11) NULL,
  `area_sqm` DECIMAL(10,2) NULL COMMENT 'Area in square meters',
  `construction_year` INT(4) NULL,
  `last_renovation` DATE NULL,
  `condition_status` ENUM('Excellent','Good','Fair','Poor','Critical') NOT NULL DEFAULT 'Good',
  `notes` TEXT NULL,
  `status` TINYINT(1) NOT NULL DEFAULT 1 COMMENT '1=Active, 0=Inactive',
  `created_by` INT(11) NULL,
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`facility_id`),
  KEY `idx_facility_type` (`facility_type`),
  KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Note: facilities_maintenance table already exists from previous schema
-- The existing structure should be sufficient for facility maintenance tracking

-- Sample data for testing
INSERT INTO `facilities` (`facility_name`, `facility_type`, `location`, `capacity`, `condition_status`) VALUES
('Main Administration Block', 'Building', 'Block A', 50, 'Good'),
('Science Laboratory 1', 'Laboratory', 'Block B, Room 201', 40, 'Good'),
('School Library', 'Library', 'Block C', 100, 'Excellent'),
('Primary Playground', 'Playground', 'East Wing', 200, 'Good'),
('School Cafeteria', 'Cafeteria', 'Block D', 300, 'Good');

-- Sample vehicles for testing
INSERT INTO `vehicles` (`vehicle_number`, `vehicle_type`, `make`, `model`, `year`, `capacity`, `fuel_type`, `status`, `assigned_driver`) VALUES
('KCA 001A', 'Bus', 'Isuzu', 'NQR 70', 2023, 51, 'Diesel', 'Active', 'John Mwangi'),
('KCB 002B', 'Van', 'Toyota', 'Hiace', 2022, 14, 'Diesel', 'Active', 'Mary Wanjiru');

-- Indexes for performance (drop first to avoid errors if they exist)
DROP INDEX IF EXISTS idx_paye_effective ON paye_tax_brackets;
CREATE INDEX idx_paye_effective ON paye_tax_brackets(effective_date, status);

DROP INDEX IF EXISTS idx_vehicle_status ON vehicles;
CREATE INDEX idx_vehicle_status ON vehicles(status, vehicle_type);

DROP INDEX IF EXISTS idx_maintenance_vehicle_date ON vehicle_maintenance;
CREATE INDEX idx_maintenance_vehicle_date ON vehicle_maintenance(vehicle_id, maintenance_date);

DROP INDEX IF EXISTS idx_facility_condition ON facilities;
CREATE INDEX idx_facility_condition ON facilities(condition_status, status);
