-- ============================================================================
-- ACCOUNTING SYSTEM - ENHANCED DATABASE TABLES
-- Created for ARM / THE KING'S SCHOOL
-- Purpose: Support comprehensive financial management, auditing, and compliance
-- ============================================================================

-- ============================================================================
-- 1. PAYMENT METHODS - Supported payment channels
-- ============================================================================
CREATE TABLE IF NOT EXISTS `payment_methods` (
  `payment_method_id` INT(11) NOT NULL AUTO_INCREMENT,
  `method_name` VARCHAR(100) NOT NULL,
  `method_code` VARCHAR(20) NOT NULL UNIQUE,
  `description` TEXT,
  `status` INT(2) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`payment_method_id`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 2. BANK ACCOUNTS - School's bank accounts
-- ============================================================================
CREATE TABLE IF NOT EXISTS `bank_accounts` (
  `bank_account_id` INT(11) NOT NULL AUTO_INCREMENT,
  `bank_name` VARCHAR(150) NOT NULL,
  `account_number` VARCHAR(50) NOT NULL,
  `account_holder` VARCHAR(150),
  `branch` VARCHAR(100),
  `currency` VARCHAR(10) DEFAULT 'KES',
  `account_type` ENUM('Current', 'Savings', 'Investment') DEFAULT 'Current',
  `balance` DECIMAL(15, 2) DEFAULT 0,
  `status` INT(2) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`bank_account_id`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 3. BANK TRANSACTIONS - All bank-related transactions
-- ============================================================================
CREATE TABLE IF NOT EXISTS `bank_transactions` (
  `transaction_id` INT(11) NOT NULL AUTO_INCREMENT,
  `bank_account_id` INT(11) NOT NULL,
  `transaction_type` ENUM('Deposit', 'Withdrawal', 'Transfer', 'Fee', 'Interest') NOT NULL,
  `amount` DECIMAL(15, 2) NOT NULL,
  `reference_number` VARCHAR(100),
  `description` TEXT,
  `transaction_date` DATE NOT NULL,
  `transaction_time` TIME,
  `balance_after` DECIMAL(15, 2),
  `related_invoice_id` INT(11),
  `related_payment_id` INT(11),
  `status` ENUM('Pending', 'Completed', 'Failed', 'Reconciled') DEFAULT 'Pending',
  `created_by` INT(11),
  `verified_by` INT(11),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`transaction_id`),
  KEY `fk_bank_account` (`bank_account_id`),
  KEY `idx_transaction_date` (`transaction_date`),
  KEY `idx_status` (`status`),
  KEY `idx_transaction_type` (`transaction_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 4. VENDORS - Vendor and supplier management
-- ============================================================================
CREATE TABLE IF NOT EXISTS `vendors` (
  `vendor_id` INT(11) NOT NULL AUTO_INCREMENT,
  `vendor_name` VARCHAR(150) NOT NULL,
  `vendor_code` VARCHAR(50) UNIQUE,
  `contact_person` VARCHAR(100),
  `email` VARCHAR(100),
  `phone` VARCHAR(20),
  `address` TEXT,
  `city` VARCHAR(100),
  `postal_code` VARCHAR(20),
  `country` VARCHAR(100),
  `tax_id` VARCHAR(50),
  `payment_terms` VARCHAR(100),
  `bank_name` VARCHAR(150),
  `bank_account` VARCHAR(50),
  `bank_code` VARCHAR(20),
  `category` VARCHAR(100),
  `status` INT(2) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`vendor_id`),
  INDEX `idx_status` (`status`),
  INDEX `idx_category` (`category`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 5. VENDOR CONTRACTS - Vendor agreements and contracts
-- ============================================================================
CREATE TABLE IF NOT EXISTS `vendor_contracts` (
  `contract_id` INT(11) NOT NULL AUTO_INCREMENT,
  `vendor_id` INT(11) NOT NULL,
  `contract_title` VARCHAR(150) NOT NULL,
  `contract_number` VARCHAR(50) UNIQUE,
  `start_date` DATE NOT NULL,
  `end_date` DATE,
  `total_contract_value` DECIMAL(15, 2),
  `contract_terms` TEXT,
  `renewal_reminder_date` DATE,
  `status` ENUM('Draft', 'Active', 'Expired', 'Terminated') DEFAULT 'Draft',
  `created_by` INT(11),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`contract_id`),
  KEY `fk_vendor` (`vendor_id`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 6. EXPENSE CATEGORIES - Categorization for expenses
-- ============================================================================
CREATE TABLE IF NOT EXISTS `expense_categories` (
  `category_id` INT(11) NOT NULL AUTO_INCREMENT,
  `category_name` VARCHAR(100) NOT NULL UNIQUE,
  `category_code` VARCHAR(20) UNIQUE,
  `description` TEXT,
  `budget_code` VARCHAR(50),
  `status` INT(2) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`category_id`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 7. ACCOUNTS PAYABLE - Vendor invoices and payables
-- ============================================================================
CREATE TABLE IF NOT EXISTS `accounts_payable` (
  `payable_id` INT(11) NOT NULL AUTO_INCREMENT,
  `vendor_id` INT(11) NOT NULL,
  `invoice_number` VARCHAR(50) UNIQUE NOT NULL,
  `invoice_date` DATE NOT NULL,
  `due_date` DATE NOT NULL,
  `amount` DECIMAL(15, 2) NOT NULL,
  `amount_paid` DECIMAL(15, 2) DEFAULT 0,
  `description` TEXT,
  `status` ENUM('Draft', 'Approved', 'Received', 'Paid', 'Cancelled') DEFAULT 'Draft',
  `payment_terms` VARCHAR(100),
  `created_by` INT(11),
  `approved_by` INT(11),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`payable_id`),
  KEY `fk_vendor` (`vendor_id`),
  INDEX `idx_status` (`status`),
  INDEX `idx_invoice_date` (`invoice_date`),
  INDEX `idx_due_date` (`due_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 8. PAYMENT SCHEDULE - Schedule for vendor payments
-- ============================================================================
CREATE TABLE IF NOT EXISTS `payment_schedule` (
  `schedule_id` INT(11) NOT NULL AUTO_INCREMENT,
  `payable_id` INT(11) NOT NULL,
  `payment_number` INT(2),
  `scheduled_date` DATE NOT NULL,
  `amount_due` DECIMAL(15, 2) NOT NULL,
  `amount_paid` DECIMAL(15, 2) DEFAULT 0,
  `payment_date` DATE,
  `payment_method_id` INT(11),
  `bank_transaction_id` INT(11),
  `status` ENUM('Pending', 'Partially Paid', 'Paid', 'Overdue', 'Cancelled') DEFAULT 'Pending',
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`schedule_id`),
  KEY `fk_payable` (`payable_id`),
  KEY `fk_payment_method` (`payment_method_id`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 9. ACCOUNTS RECEIVABLE - School receivables (fees, donations, etc)
-- ============================================================================
CREATE TABLE IF NOT EXISTS `accounts_receivable` (
  `receivable_id` INT(11) NOT NULL AUTO_INCREMENT,
  `student_id` INT(11),
  `parent_id` INT(11),
  `invoice_number` VARCHAR(50) UNIQUE NOT NULL,
  `invoice_date` DATE NOT NULL,
  `due_date` DATE NOT NULL,
  `invoice_type` ENUM('Fees', 'Donation', 'Event', 'Other') DEFAULT 'Fees',
  `amount` DECIMAL(15, 2) NOT NULL,
  `amount_paid` DECIMAL(15, 2) DEFAULT 0,
  `discount_applied` DECIMAL(15, 2) DEFAULT 0,
  `description` TEXT,
  `status` ENUM('Draft', 'Sent', 'Partial', 'Paid', 'Overdue', 'Cancelled') DEFAULT 'Draft',
  `payment_terms` VARCHAR(100),
  `created_by` INT(11),
  `sent_date` DATE,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`receivable_id`),
  INDEX `idx_student` (`student_id`),
  INDEX `idx_status` (`status`),
  INDEX `idx_invoice_date` (`invoice_date`),
  INDEX `idx_due_date` (`due_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 10. FEE STRUCTURES - Template-based fee configurations
-- ============================================================================
CREATE TABLE IF NOT EXISTS `fee_structures` (
  `fee_structure_id` INT(11) NOT NULL AUTO_INCREMENT,
  `structure_name` VARCHAR(100) NOT NULL,
  `academic_year_id` INT(11),
  `term_id` INT(11),
  `class_id` INT(11),
  `stream_id` INT(11),
  `base_tuition_fee` DECIMAL(15, 2),
  `boarding_fee` DECIMAL(15, 2),
  `activity_fee` DECIMAL(15, 2),
  `examination_fee` DECIMAL(15, 2),
  `technology_fee` DECIMAL(15, 2),
  `sports_fee` DECIMAL(15, 2),
  `other_fees` DECIMAL(15, 2),
  `total_fee` DECIMAL(15, 2),
  `description` TEXT,
  `status` INT(2) DEFAULT 1,
  `created_by` INT(11),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`fee_structure_id`),
  INDEX `idx_academic_year` (`academic_year_id`),
  INDEX `idx_term` (`term_id`),
  INDEX `idx_class` (`class_id`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 11. SCHOLARSHIPS & DISCOUNTS - Student discounts and scholarships
-- ============================================================================
CREATE TABLE IF NOT EXISTS `scholarships_discounts` (
  `discount_id` INT(11) NOT NULL AUTO_INCREMENT,
  `student_id` INT(11) NOT NULL,
  `discount_type` ENUM('Scholarship', 'Sibling', 'Merit', 'Financial Hardship', 'Staff Child', 'Other') DEFAULT 'Other',
  `discount_amount` DECIMAL(15, 2),
  `discount_percentage` DECIMAL(5, 2),
  `academic_year_id` INT(11),
  `start_date` DATE NOT NULL,
  `end_date` DATE,
  `reason` TEXT,
  `approval_status` ENUM('Pending', 'Approved', 'Rejected', 'Cancelled') DEFAULT 'Pending',
  `approved_by` INT(11),
  `status` INT(2) DEFAULT 1,
  `created_by` INT(11),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`discount_id`),
  KEY `fk_student` (`student_id`),
  INDEX `idx_status` (`status`),
  INDEX `idx_approval_status` (`approval_status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 12. BUDGETS - Budget planning and tracking
-- ============================================================================
CREATE TABLE IF NOT EXISTS `budgets` (
  `budget_id` INT(11) NOT NULL AUTO_INCREMENT,
  `budget_name` VARCHAR(150) NOT NULL,
  `department` VARCHAR(100),
  `fiscal_year` INT(4),
  `budget_period` VARCHAR(50),
  `total_budget` DECIMAL(15, 2) NOT NULL,
  `allocated_amount` DECIMAL(15, 2),
  `spent_amount` DECIMAL(15, 2) DEFAULT 0,
  `remaining_amount` DECIMAL(15, 2),
  `status` ENUM('Draft', 'Submitted', 'Approved', 'Active', 'Closed') DEFAULT 'Draft',
  `created_by` INT(11),
  `approved_by` INT(11),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`budget_id`),
  INDEX `idx_fiscal_year` (`fiscal_year`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 13. BUDGET LINES - Individual budget line items
-- ============================================================================
CREATE TABLE IF NOT EXISTS `budget_lines` (
  `budget_line_id` INT(11) NOT NULL AUTO_INCREMENT,
  `budget_id` INT(11) NOT NULL,
  `category_id` INT(11),
  `line_description` VARCHAR(200) NOT NULL,
  `budgeted_amount` DECIMAL(15, 2) NOT NULL,
  `actual_amount` DECIMAL(15, 2) DEFAULT 0,
  `variance` DECIMAL(15, 2),
  `percentage_spent` DECIMAL(5, 2),
  `notes` TEXT,
  `status` INT(2) DEFAULT 1,
  PRIMARY KEY (`budget_line_id`),
  KEY `fk_budget` (`budget_id`),
  KEY `fk_category` (`category_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 14. PAYROLL - Enhanced salary records with tax and deductions
-- ============================================================================
CREATE TABLE IF NOT EXISTS `payroll_records` (
  `payroll_id` INT(11) NOT NULL AUTO_INCREMENT,
  `staff_id` INT(11) NOT NULL,
  `payroll_period` VARCHAR(50),
  `salary_month` INT(2),
  `salary_year` INT(4),
  `base_salary` DECIMAL(15, 2),
  `allowances` DECIMAL(15, 2) DEFAULT 0,
  `bonuses` DECIMAL(15, 2) DEFAULT 0,
  `gross_salary` DECIMAL(15, 2),
  `paye_tax` DECIMAL(15, 2) DEFAULT 0,
  `social_security` DECIMAL(15, 2) DEFAULT 0,
  `health_insurance` DECIMAL(15, 2) DEFAULT 0,
  `other_deductions` DECIMAL(15, 2) DEFAULT 0,
  `total_deductions` DECIMAL(15, 2),
  `net_salary` DECIMAL(15, 2),
  `payment_method_id` INT(11),
  `payment_date` DATE,
  `payment_status` ENUM('Draft', 'Approved', 'Paid', 'Cancelled') DEFAULT 'Draft',
  `processed_by` INT(11),
  `payment_reference` VARCHAR(100),
  `notes` TEXT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`payroll_id`),
  KEY `fk_staff` (`staff_id`),
  KEY `fk_payment_method` (`payment_method_id`),
  INDEX `idx_period` (`salary_month`, `salary_year`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 15. TAX CONFIGURATION - Tax rates and rules
-- ============================================================================
CREATE TABLE IF NOT EXISTS `tax_configuration` (
  `tax_id` INT(11) NOT NULL AUTO_INCREMENT,
  `tax_type` VARCHAR(50) UNIQUE NOT NULL,
  `description` TEXT,
  `tax_year` INT(4),
  `tax_bracket_1_from` DECIMAL(15, 2),
  `tax_bracket_1_to` DECIMAL(15, 2),
  `tax_bracket_1_rate` DECIMAL(5, 2),
  `tax_bracket_2_from` DECIMAL(15, 2),
  `tax_bracket_2_to` DECIMAL(15, 2),
  `tax_bracket_2_rate` DECIMAL(5, 2),
  `tax_bracket_3_from` DECIMAL(15, 2),
  `tax_bracket_3_to` DECIMAL(15, 2),
  `tax_bracket_3_rate` DECIMAL(5, 2),
  `tax_bracket_4_from` DECIMAL(15, 2),
  `tax_bracket_4_to` DECIMAL(15, 2),
  `tax_bracket_4_rate` DECIMAL(5, 2),
  `status` INT(2) DEFAULT 1,
  PRIMARY KEY (`tax_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 16. TRANSACTION LOGS - Audit trail for all financial transactions
-- ============================================================================
CREATE TABLE IF NOT EXISTS `transaction_logs` (
  `log_id` INT(11) NOT NULL AUTO_INCREMENT,
  `transaction_type` VARCHAR(100) NOT NULL,
  `related_module` VARCHAR(50),
  `related_id` INT(11),
  `user_id` INT(11),
  `username` VARCHAR(100),
  `action` VARCHAR(50),
  `old_value` TEXT,
  `new_value` TEXT,
  `amount` DECIMAL(15, 2),
  `description` TEXT,
  `ip_address` VARCHAR(45),
  `user_agent` TEXT,
  `log_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`log_id`),
  KEY `idx_user` (`user_id`),
  KEY `idx_type` (`transaction_type`),
  KEY `idx_log_date` (`log_date`),
  KEY `idx_related` (`related_module`, `related_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 17. AUDIT TRAIL - Compliance and audit records
-- ============================================================================
CREATE TABLE IF NOT EXISTS `audit_trail` (
  `audit_id` INT(11) NOT NULL AUTO_INCREMENT,
  `audit_type` VARCHAR(100) NOT NULL,
  `audit_category` VARCHAR(50),
  `description` TEXT,
  `findings` TEXT,
  `audit_date` DATE NOT NULL,
  `auditor_id` INT(11),
  `auditor_name` VARCHAR(100),
  `status` ENUM('Scheduled', 'In Progress', 'Completed', 'Cleared') DEFAULT 'Scheduled',
  `result` ENUM('Pass', 'Fail', 'Conditional') DEFAULT 'Pass',
  `recommendations` TEXT,
  `follow_up_date` DATE,
  `follow_up_notes` TEXT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`audit_id`),
  INDEX `idx_audit_type` (`audit_type`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 18. FINANCIAL REPORTS CACHE - Store generated financial reports
-- ============================================================================
CREATE TABLE IF NOT EXISTS `financial_reports` (
  `report_id` INT(11) NOT NULL AUTO_INCREMENT,
  `report_type` VARCHAR(100) NOT NULL,
  `report_name` VARCHAR(150),
  `report_period_from` DATE,
  `report_period_to` DATE,
  `report_content` LONGBLOB,
  `report_format` ENUM('PDF', 'Excel', 'HTML', 'JSON') DEFAULT 'PDF',
  `summary_data` JSON,
  `generated_by` INT(11),
  `generated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`report_id`),
  INDEX `idx_report_type` (`report_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 19. INVOICE TEMPLATES - Email and print templates for invoices
-- ============================================================================
CREATE TABLE IF NOT EXISTS `invoice_templates` (
  `template_id` INT(11) NOT NULL AUTO_INCREMENT,
  `template_name` VARCHAR(100) NOT NULL,
  `template_type` VARCHAR(50),
  `template_content` TEXT NOT NULL,
  `subject` VARCHAR(200),
  `is_default` INT(2) DEFAULT 0,
  `status` INT(2) DEFAULT 1,
  `created_by` INT(11),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`template_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 20. ASSET MANAGEMENT - School assets tracking
-- ============================================================================
CREATE TABLE IF NOT EXISTS `assets` (
  `asset_id` INT(11) NOT NULL AUTO_INCREMENT,
  `asset_name` VARCHAR(150) NOT NULL,
  `asset_code` VARCHAR(50) UNIQUE,
  `asset_type` VARCHAR(100),
  `category` VARCHAR(100),
  `description` TEXT,
  `purchase_date` DATE,
  `purchase_price` DECIMAL(15, 2),
  `depreciation_method` VARCHAR(50),
  `depreciation_rate` DECIMAL(5, 2),
  `current_value` DECIMAL(15, 2),
  `location` VARCHAR(100),
  `responsible_person` INT(11),
  `warranty_expiry` DATE,
  `maintenance_schedule` VARCHAR(50),
  `status` ENUM('Active', 'Maintenance', 'Disposed', 'Inactive') DEFAULT 'Active',
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`asset_id`),
  INDEX `idx_asset_type` (`asset_type`),
  INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 21. STOCK MANAGEMENT - Inventory stock levels
-- ============================================================================
CREATE TABLE IF NOT EXISTS `stock_management` (
  `stock_id` INT(11) NOT NULL AUTO_INCREMENT,
  `item_id` INT(11),
  `stock_in` INT(11) DEFAULT 0,
  `stock_out` INT(11) DEFAULT 0,
  `current_stock` INT(11),
  `reorder_level` INT(11),
  `status_date` DATE,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`stock_id`),
  INDEX `idx_item` (`item_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 22. STOCK REQUISITIONS - Staff/teacher stock requests
-- ============================================================================
CREATE TABLE IF NOT EXISTS `stock_requisitions` (
  `requisition_id` INT(11) NOT NULL AUTO_INCREMENT,
  `requisition_number` VARCHAR(50) UNIQUE,
  `requested_by` INT(11),
  `requested_date` DATE,
  `requisition_items` JSON,
  `total_items` INT(11),
  `purpose` TEXT,
  `approval_status` ENUM('Pending', 'Approved', 'Rejected', 'Fulfilled') DEFAULT 'Pending',
  `approved_by` INT(11),
  `approval_date` DATE,
  `fulfilled_date` DATE,
  `notes` TEXT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`requisition_id`),
  INDEX `idx_status` (`approval_status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 23. FACILITIES MAINTENANCE - Maintenance scheduling and tracking
-- ============================================================================
CREATE TABLE IF NOT EXISTS `facilities_maintenance` (
  `maintenance_id` INT(11) NOT NULL AUTO_INCREMENT,
  `facility_name` VARCHAR(150) NOT NULL,
  `maintenance_type` VARCHAR(100),
  `scheduled_date` DATE,
  `completion_date` DATE,
  `description` TEXT,
  `cost` DECIMAL(15, 2),
  `vendor_id` INT(11),
  `priority` ENUM('Low', 'Medium', 'High', 'Critical') DEFAULT 'Medium',
  `status` ENUM('Scheduled', 'In Progress', 'Completed', 'Cancelled') DEFAULT 'Scheduled',
  `assigned_to` INT(11),
  `notes` TEXT,
  `created_by` INT(11),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`maintenance_id`),
  INDEX `idx_status` (`status`),
  INDEX `idx_priority` (`priority`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 24. USER ACCESS LOG - Track user login and activities for security
-- ============================================================================
CREATE TABLE IF NOT EXISTS `user_access_log` (
  `access_id` INT(11) NOT NULL AUTO_INCREMENT,
  `user_id` INT(11),
  `username` VARCHAR(100),
  `action` VARCHAR(50),
  `access_module` VARCHAR(100),
  `ip_address` VARCHAR(45),
  `user_agent` TEXT,
  `status` VARCHAR(20),
  `access_time` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`access_id`),
  KEY `idx_user` (`user_id`),
  KEY `idx_access_time` (`access_time`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- 25. SYSTEM CONFIGURATION - System settings and parameters
-- ============================================================================
CREATE TABLE IF NOT EXISTS `system_configuration` (
  `config_id` INT(11) NOT NULL AUTO_INCREMENT,
  `config_key` VARCHAR(100) UNIQUE NOT NULL,
  `config_value` TEXT,
  `data_type` VARCHAR(20),
  `description` TEXT,
  `category` VARCHAR(50),
  `editable` INT(2) DEFAULT 1,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`config_id`),
  INDEX `idx_category` (`category`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- Insert default payment methods
-- ============================================================================
INSERT INTO `payment_methods` (`method_name`, `method_code`, `description`, `status`) VALUES
('Bank Transfer', 'BANK_TRANSFER', 'Direct bank transfer/EFT', 1),
('Check', 'CHECK', 'Payment by cheque', 1),
('Cash', 'CASH', 'Direct cash payment', 1),
('Credit Card', 'CREDIT_CARD', 'Credit card payment', 1),
('Mobile Money', 'MOBILE_MONEY', 'Mobile money transfer (M-Pesa, etc)', 1),
('Standing Order', 'STANDING_ORDER', 'Recurring bank standing order', 1)
ON DUPLICATE KEY UPDATE `status` = 1;

-- ============================================================================
-- Insert default expense categories
-- ============================================================================
INSERT INTO `expense_categories` (`category_name`, `category_code`, `description`) VALUES
('Salaries & Wages', 'SALARY', 'Staff and teacher salaries'),
('Utilities', 'UTILITIES', 'Electricity, water, gas bills'),
('Maintenance & Repairs', 'MAINTENANCE', 'Building and equipment maintenance'),
('Supplies & Materials', 'SUPPLIES', 'Educational and office supplies'),
('Transportation', 'TRANSPORT', 'School bus and transportation costs'),
('Marketing & Advertising', 'MARKETING', 'Promotional and advertising expenses'),
('Professional Services', 'PROFESSIONAL', 'Consulting, legal, audit services'),
('Insurance', 'INSURANCE', 'Insurance premiums'),
('Catering', 'CATERING', 'Food and catering services'),
('Equipment & Technology', 'EQUIPMENT', 'Computers, furniture, equipment'),
('Training & Development', 'TRAINING', 'Staff training and professional development'),
('Other Operating Expenses', 'OTHER', 'Miscellaneous operating expenses')
ON DUPLICATE KEY UPDATE `category_code` = VALUES(`category_code`);

-- ============================================================================
-- Insert default tax configuration for Kenya
-- ============================================================================
INSERT INTO `tax_configuration` (`tax_type`, `description`, `tax_year`, `tax_bracket_1_from`, `tax_bracket_1_to`, `tax_bracket_1_rate`, `tax_bracket_2_from`, `tax_bracket_2_to`, `tax_bracket_2_rate`, `tax_bracket_3_from`, `tax_bracket_3_to`, `tax_bracket_3_rate`, `tax_bracket_4_from`, `tax_bracket_4_to`, `tax_bracket_4_rate`, `status`) VALUES
('PAYE', 'Personal Income Tax - Kenya', 2024, 0, 24000, 10, 24001, 40625, 15, 40626, 57292, 20, 57293, 9999999, 25, 1)
ON DUPLICATE KEY UPDATE `status` = 1;

-- ============================================================================
-- Insert default system configuration
-- ============================================================================
INSERT INTO `system_configuration` (`config_key`, `config_value`, `data_type`, `description`, `category`) VALUES
('currency', 'KES', 'string', 'Default currency code', 'General'),
('currency_symbol', 'KES', 'string', 'Currency symbol for display', 'General'),
('decimal_places', '2', 'integer', 'Decimal places for amounts', 'General'),
('date_format', 'd/M/Y', 'string', 'Date display format', 'General'),
('enable_audit_log', 'true', 'boolean', 'Enable transaction logging', 'Security'),
('enable_data_encryption', 'false', 'boolean', 'Enable data encryption for sensitive fields', 'Security'),
('backup_frequency', 'weekly', 'string', 'Database backup frequency', 'Backup'),
('backup_retention_days', '90', 'integer', 'Number of days to keep backups', 'Backup'),
('financial_year_start', '01', 'string', 'Month when financial year starts', 'Financial'),
('tax_calculation_method', 'automatic', 'string', 'How to calculate taxes', 'Financial'),
('invoice_prefix', 'INV-', 'string', 'Prefix for generated invoices', 'Financial'),
('receipt_prefix', 'RCP-', 'string', 'Prefix for generated receipts', 'Financial')
ON DUPLICATE KEY UPDATE `config_value` = VALUES(`config_value`);

-- ============================================================================
-- END OF ACCOUNTING SYSTEM DATABASE SCHEMA
-- ============================================================================
