<?php
/**
 * Centralized Role Management Functions
 * This file contains functions to handle user roles consistently across the system
 * Enhanced with financial management and permission checking functions
 */

/**
 * Get role name by level
 * @param int $level The role level
 * @return string The role name
 */
function getRoleName($level) {
    $roles = [
        1 => 'Super Admin',
        2 => 'Principal', 
        3 => 'Accounts Management',
        4 => 'Head of Section',
        5 => 'Librarian',
        6 => 'Secretary',
        7 => 'Head Teacher',
        8 => 'Finance Manager',
        9 => 'HR Admin Director',
        10 => 'Accountant'
    ];
    
    return $roles[$level] ?? 'User';
}

/**
 * Get all available roles for dropdowns
 * @return array Array of roles [level => name]
 */
function getAllRoles() {
    return [
        1 => 'Super Admin',
        2 => 'Principal',
        3 => 'Accounts Management', 
        4 => 'Head of Section',
        5 => 'Librarian',
        6 => 'Secretary',
        7 => 'Head Teacher',
        8 => 'Finance Manager',
        9 => 'HR Admin Director',
        10 => 'Accountant'
    ];
}

/**
 * Generate role option elements for select dropdowns
 * @param int $selected_level The currently selected role level
 * @param array $exclude_levels Array of role levels to exclude
 * @return string HTML option elements
 */
function generateRoleOptions($selected_level = 0, $exclude_levels = []) {
    $roles = getAllRoles();
    $options = '';
    
    foreach ($roles as $level => $name) {
        if (in_array($level, $exclude_levels)) continue;
        
        $selected = ($level == $selected_level) ? 'selected' : '';
        $options .= "<option value=\"{$level}\" {$selected}>" . htmlspecialchars($name) . "</option>\n";
    }
    
    return $options;
}

/**
 * Check if a role level is valid
 * @param int $level The role level to check
 * @return bool True if valid, false otherwise
 */
function isValidRole($level) {
    $roles = getAllRoles();
    return isset($roles[$level]);
}

/**
 * Check if user has specific role
 * @param int $user_role The user's role level
 * @param int|array $required_roles The required role(s)
 * @return bool True if user has the role
 */
function hasRole($user_role, $required_roles) {
    if (is_array($required_roles)) {
        return in_array($user_role, $required_roles);
    }
    return $user_role == $required_roles;
}

/**
 * Check if user has financial access (Finance Manager, HR Admin Director, Accountant, or Super Admin/Principal)
 * @param int $user_role The user's role level
 * @return bool True if user has financial access
 */
function hasFinancialAccess($user_role) {
    $financial_roles = [1, 2, 3, 8, 9, 10]; // Super Admin, Principal, Accounts Management, Finance Manager, HR Admin Director, Accountant
    return in_array($user_role, $financial_roles);
}

/**
 * Check if user has full financial permissions (Super Admin, Principal, Finance Manager, HR Admin Director)
 * @param int $user_role The user's role level
 * @return bool True if user has full financial permissions
 */
function hasFullFinancialPermissions($user_role) {
    $full_perms_roles = [1, 2, 8, 9]; // Super Admin, Principal, Finance Manager, HR Admin Director
    return in_array($user_role, $full_perms_roles);
}

/**
 * Check if user has vendor management access
 * @param int $user_role The user's role level
 * @return bool True if user can manage vendors
 */
function canManageVendors($user_role) {
    $vendor_mgmt_roles = [1, 2, 8, 9, 10]; // Super Admin, Principal, Finance Manager, HR Admin Director, Accountant
    return in_array($user_role, $vendor_mgmt_roles);
}

/**
 * Check if user can approve payments
 * @param int $user_role The user's role level
 * @return bool True if user can approve payments
 */
function canApprovePayments($user_role) {
    $approval_roles = [1, 2, 8, 9]; // Super Admin, Principal, Finance Manager, HR Admin Director
    return in_array($user_role, $approval_roles);
}

/**
 * Check if user can view financial reports
 * @param int $user_role The user's role level
 * @return bool True if user can view reports
 */
function canViewFinancialReports($user_role) {
    $report_roles = [1, 2, 3, 8, 9, 10]; // Super Admin, Principal, Accounts Management, Finance Manager, HR Admin Director, Accountant
    return in_array($user_role, $report_roles);
}

/**
 * Check if user can manage budgets
 * @param int $user_role The user's role level
 * @return bool True if user can manage budgets
 */
function canManageBudgets($user_role) {
    $budget_roles = [1, 2, 8, 9, 10]; // Super Admin, Principal, Finance Manager, HR Admin Director, Accountant
    return in_array($user_role, $budget_roles);
}

/**
 * Get financial permissions array for a specific role
 * @param int $user_role The user's role level
 * @return array Array of permissions
 */
function getFinancialPermissions($user_role) {
    $permissions = [
        'view_financial_reports' => hasRole($user_role, [1, 2, 3, 8, 9, 10]),
        'manage_vendors' => canManageVendors($user_role),
        'manage_budgets' => canManageBudgets($user_role),
        'approve_payments' => canApprovePayments($user_role),
        'manage_invoices' => hasRole($user_role, [1, 2, 3, 8, 9, 10]),
        'manage_fees' => hasRole($user_role, [1, 2, 3, 8, 9, 10]),
        'manage_payroll' => hasRole($user_role, [1, 2, 8, 9]),
        'audit_access' => hasRole($user_role, [1, 2, 8, 9]),
        'system_configuration' => hasRole($user_role, [1]),
        'view_audit_trail' => hasRole($user_role, [1, 2, 8, 9]),
        'export_reports' => hasRole($user_role, [1, 2, 3, 8, 9, 10])
    ];
    return $permissions;
}
?>
