<?php
/**
 * PAYE Tax Calculator
 * Calculates Kenya PAYE tax based on current tax brackets in the database
 * Uses progressive taxation system
 */

if (!function_exists('calculatePAYE')) {
    /**
     * Calculate PAYE tax for a given taxable income
     * @param float $taxableIncome The monthly taxable income
     * @param mysqli $con Database connection
     * @return array ['paye' => float, 'breakdown' => array, 'error' => string|null]
     */
    function calculatePAYE($taxableIncome, $con) {
        $result = [
            'paye' => 0.00,
            'breakdown' => [],
            'error' => null
        ];
        
        if ($taxableIncome <= 0) {
            return $result;
        }
        
        // Fetch active tax brackets ordered by minimum amount
        $query = "SELECT * FROM paye_tax_brackets 
                  WHERE status = 1 
                  ORDER BY min_amount ASC";
        $brackets_result = mysqli_query($con, $query);
        
        if (!$brackets_result) {
            $result['error'] = 'Failed to fetch tax brackets';
            return $result;
        }
        
        if (mysqli_num_rows($brackets_result) === 0) {
            $result['error'] = 'No active tax brackets configured. Please configure PAYE tax brackets.';
            return $result;
        }
        
        $totalTax = 0.00;
        $remainingIncome = $taxableIncome;
        
        while ($bracket = mysqli_fetch_assoc($brackets_result)) {
            $min = (float) $bracket['min_amount'];
            $max = $bracket['max_amount'] ? (float) $bracket['max_amount'] : PHP_FLOAT_MAX;
            $rate = (float) $bracket['rate'];
            
            // Skip if income doesn't reach this bracket
            if ($taxableIncome <= $min) {
                continue;
            }
            
            // Calculate taxable amount in this bracket
            $bracketMin = max($min, 0);
            $bracketMax = min($max, $taxableIncome);
            $taxableInBracket = $bracketMax - $bracketMin;
            
            if ($taxableInBracket > 0) {
                $taxInBracket = $taxableInBracket * ($rate / 100);
                $totalTax += $taxInBracket;
                
                $result['breakdown'][] = [
                    'bracket_name' => $bracket['bracket_name'],
                    'min' => $min,
                    'max' => $bracket['max_amount'],
                    'rate' => $rate,
                    'taxable_amount' => $taxableInBracket,
                    'tax' => $taxInBracket
                ];
            }
        }
        
        $result['paye'] = round($totalTax, 2);
        return $result;
    }
}

/**
 * Get all active tax brackets for display/reference
 * @param mysqli $con Database connection
 * @return array Array of tax bracket objects
 */
function getActiveTaxBrackets($con) {
    // Return the latest active bracket per bracket_name to avoid duplicates
    $query = "SELECT p.* FROM paye_tax_brackets p
              WHERE p.bracket_id IN (
                  SELECT MAX(bracket_id) FROM paye_tax_brackets WHERE status=1 GROUP BY bracket_name
              )
              ORDER BY p.min_amount ASC";
    $result = mysqli_query($con, $query);
    
    $brackets = [];
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $brackets[] = $row;
        }
    }
    return $brackets;
}

/**
 * Get tax brackets as JSON for JavaScript use
 * @param mysqli $con Database connection
 * @return string JSON encoded tax brackets
 */
function getTaxBracketsJSON($con) {
    $brackets = getActiveTaxBrackets($con);
    return json_encode($brackets);
}

/**
 * Calculate net salary after all deductions
 * @param float $grossSalary
 * @param float $nssf
 * @param float $nhif Optional health insurance
 * @param float $housingLevy Optional housing levy
 * @param mysqli $con Database connection
 * @return array ['taxable_income', 'paye', 'net_salary', 'breakdown']
 */
function calculateNetSalary($grossSalary, $nssf, $con, $nhif = 0, $housingLevy = 0) {
    // Calculate taxable income (Gross - NSSF - other pre-tax deductions)
    $taxableIncome = $grossSalary - $nssf;
    
    // Calculate PAYE
    $payeResult = calculatePAYE($taxableIncome, $con);
    
    // Calculate net salary
    $netSalary = $grossSalary - $nssf - $payeResult['paye'] - $nhif - $housingLevy;
    
    return [
        'gross_salary' => $grossSalary,
        'nssf' => $nssf,
        'nhif' => $nhif,
        'housing_levy' => $housingLevy,
        'taxable_income' => $taxableIncome,
        'paye' => $payeResult['paye'],
        'paye_breakdown' => $payeResult['breakdown'],
        'net_salary' => round($netSalary, 2),
        'error' => $payeResult['error']
    ];
}
?>
