<?php
/**
 * Financial Transaction Logging and Audit Trail Functions
 * Used for compliance, auditing, and transaction tracking
 */

/**
 * Log a financial transaction to the transaction_logs table
 * @param mysqli $con Database connection
 * @param string $transaction_type Type of transaction (e.g., 'PAYMENT', 'INVOICE', 'BUDGET')
 * @param string $related_module The module it relates to
 * @param int $related_id ID in the related module
 * @param int $user_id User performing the action
 * @param string $action Action performed (CREATE, UPDATE, DELETE, APPROVE, REJECT)
 * @param mixed $amount Amount involved
 * @param string $description Description of transaction
 * @param mixed $old_value Previous value (optional)
 * @param mixed $new_value New value (optional)
 * @return bool Success status
 */
function logTransaction($con, $transaction_type, $related_module, $related_id, $user_id, $action, $amount, $description, $old_value = null, $new_value = null) {
    try {
        $username = $_SESSION['username'] ?? 'System';
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
        
        $transaction_type = mysqli_real_escape_string($con, $transaction_type);
        $related_module = mysqli_real_escape_string($con, $related_module);
        $action = mysqli_real_escape_string($con, $action);
        $description = mysqli_real_escape_string($con, $description);
        $username = mysqli_real_escape_string($con, $username);
        $old_value = $old_value !== null ? mysqli_real_escape_string($con, (string)$old_value) : null;
        $new_value = $new_value !== null ? mysqli_real_escape_string($con, (string)$new_value) : null;
        
        $query = "INSERT INTO transaction_logs 
                 (transaction_type, related_module, related_id, user_id, username, action, 
                  old_value, new_value, amount, description, ip_address, user_agent) 
                 VALUES ('$transaction_type', '$related_module', " . (int)$related_id . ", " . (int)$user_id . ", 
                 '$username', '$action', " . ($old_value !== null ? "'$old_value'" : "NULL") . ", 
                 " . ($new_value !== null ? "'$new_value'" : "NULL") . ", " . floatval($amount) . ", 
                 '$description', '$ip_address', '$user_agent')";
        
        return mysqli_query($con, $query);
    } catch (Exception $e) {
        error_log('Error logging transaction: ' . $e->getMessage());
        return false;
    }
}

/**
 * Record an audit trail entry
 * @param mysqli $con Database connection
 * @param string $audit_type Type of audit (FINANCIAL_REVIEW, COMPLIANCE_CHECK, etc)
 * @param string $audit_category Category of audit
 * @param string $description Description
 * @param string $findings Findings
 * @param int $auditor_id ID of person conducting audit
 * @param string $status Status (Scheduled, In Progress, Completed)
 * @param string $result Result (Pass, Fail, Conditional)
 * @return bool Success status
 */
function createAuditTrail($con, $audit_type, $audit_category, $description, $findings, $auditor_id, $status = 'Scheduled', $result = 'Pass') {
    try {
        $audit_type = mysqli_real_escape_string($con, $audit_type);
        $audit_category = mysqli_real_escape_string($con, $audit_category);
        $description = mysqli_real_escape_string($con, $description);
        $findings = mysqli_real_escape_string($con, $findings);
        $status = mysqli_real_escape_string($con, $status);
        $result = mysqli_real_escape_string($con, $result);
        
        // Get auditor name
        $auditor_query = "SELECT username FROM users WHERE user_id = " . (int)$auditor_id;
        $auditor_result = mysqli_query($con, $auditor_query);
        $auditor_name = ($auditor_result && mysqli_num_rows($auditor_result) > 0) ? 
                        mysqli_fetch_array($auditor_result)['username'] : 'Unknown';
        $auditor_name = mysqli_real_escape_string($con, $auditor_name);
        
        $query = "INSERT INTO audit_trail 
                 (audit_type, audit_category, description, findings, audit_date, auditor_id, auditor_name, status, result) 
                 VALUES ('$audit_type', '$audit_category', '$description', '$findings', CURDATE(), " . (int)$auditor_id . ", 
                 '$auditor_name', '$status', '$result')";
        
        return mysqli_query($con, $query);
    } catch (Exception $e) {
        error_log('Error creating audit trail: ' . $e->getMessage());
        return false;
    }
}

/**
 * Log user access/activity for security tracking
 * @param mysqli $con Database connection
 * @param int $user_id User ID
 * @param string $action Action (LOGIN, LOGOUT, VIEW, EDIT, DELETE, EXPORT)
 * @param string $access_module Module accessed
 * @param string $status Status (SUCCESS, FAILED, UNAUTHORIZED)
 * @return bool Success status
 */
function logUserAccess($con, $user_id, $action, $access_module, $status = 'SUCCESS') {
    try {
        $username = $_SESSION['username'] ?? 'Unknown';
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $user_agent = substr($_SERVER['HTTP_USER_AGENT'] ?? 'unknown', 0, 255);
        
        $username = mysqli_real_escape_string($con, $username);
        $action = mysqli_real_escape_string($con, $action);
        $access_module = mysqli_real_escape_string($con, $access_module);
        $status = mysqli_real_escape_string($con, $status);
        
        $query = "INSERT INTO user_access_log 
                 (user_id, username, action, access_module, ip_address, user_agent, status) 
                 VALUES (" . (int)$user_id . ", '$username', '$action', '$access_module', '$ip_address', '$user_agent', '$status')";
        
        return mysqli_query($con, $query);
    } catch (Exception $e) {
        error_log('Error logging user access: ' . $e->getMessage());
        return false;
    }
}

/**
 * Get transaction history for a specific module/ID
 * @param mysqli $con Database connection
 * @param string $module Module name
 * @param int $related_id ID in that module
 * @param int $limit Limit results
 * @return array Array of transaction records
 */
function getTransactionHistory($con, $module, $related_id, $limit = 50) {
    $module = mysqli_real_escape_string($con, $module);
    $query = "SELECT * FROM transaction_logs 
              WHERE related_module = '$module' AND related_id = " . (int)$related_id . " 
              ORDER BY log_date DESC LIMIT " . (int)$limit;
    
    $result = mysqli_query($con, $query);
    $transactions = [];
    
    if ($result && mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) {
            $transactions[] = $row;
        }
    }
    
    return $transactions;
}

/**
 * Get audit trail records
 * @param mysqli $con Database connection
 * @param int $limit Limit results
 * @param string $audit_type Optional filter by audit type
 * @return array Array of audit records
 */
function getAuditTrails($con, $limit = 50, $audit_type = '') {
    $query = "SELECT * FROM audit_trail WHERE 1=1";
    
    if (!empty($audit_type)) {
        $audit_type = mysqli_real_escape_string($con, $audit_type);
        $query .= " AND audit_type = '$audit_type'";
    }
    
    $query .= " ORDER BY audit_date DESC, audit_id DESC LIMIT " . (int)$limit;
    
    $result = mysqli_query($con, $query);
    $audits = [];
    
    if ($result && mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) {
            $audits[] = $row;
        }
    }
    
    return $audits;
}

/**
 * Get user access logs
 * @param mysqli $con Database connection
 * @param int $user_id Optional filter by user
 * @param int $limit Limit results
 * @return array Array of access log records
 */
function getUserAccessLogs($con, $user_id = 0, $limit = 100) {
    $query = "SELECT * FROM user_access_log WHERE 1=1";
    
    if ($user_id > 0) {
        $query .= " AND user_id = " . (int)$user_id;
    }
    
    $query .= " ORDER BY access_time DESC LIMIT " . (int)$limit;
    
    $result = mysqli_query($con, $query);
    $logs = [];
    
    if ($result && mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) {
            $logs[] = $row;
        }
    }
    
    return $logs;
}

/**
 * Generate a financial audit report for a period
 * @param mysqli $con Database connection
 * @param string $start_date Start date (YYYY-MM-DD)
 * @param string $end_date End date (YYYY-MM-DD)
 * @return array Summary of transactions
 */
function generateFinancialAuditReport($con, $start_date, $end_date) {
    $start_date = mysqli_real_escape_string($con, $start_date);
    $end_date = mysqli_real_escape_string($con, $end_date);
    
    $report = [
        'period' => "$start_date to $end_date",
        'generated_at' => date('Y-m-d H:i:s'),
        'total_transactions' => 0,
        'by_type' => [],
        'by_action' => [],
        'total_amount_processed' => 0,
        'transactions' => []
    ];
    
    $query = "SELECT * FROM transaction_logs 
              WHERE DATE(log_date) >= '$start_date' AND DATE(log_date) <= '$end_date' 
              ORDER BY log_date DESC";
    
    $result = mysqli_query($con, $query);
    
    if ($result && mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) {
            $report['total_transactions']++;
            $report['total_amount_processed'] += $row['amount'];
            
            // Count by type
            $type = $row['transaction_type'];
            $report['by_type'][$type] = ($report['by_type'][$type] ?? 0) + 1;
            
            // Count by action
            $action = $row['action'];
            $report['by_action'][$action] = ($report['by_action'][$action] ?? 0) + 1;
            
            $report['transactions'][] = $row;
        }
    }
    
    return $report;
}

/**
 * Export transaction logs to CSV
 * @param mysqli $con Database connection
 * @param string $start_date Start date
 * @param string $end_date End date
 * @param string $filename Output filename
 */
function exportTransactionLogsToCSV($con, $start_date, $end_date, $filename = 'transaction_logs.csv') {
    $start_date = mysqli_real_escape_string($con, $start_date);
    $end_date = mysqli_real_escape_string($con, $end_date);
    
    $query = "SELECT * FROM transaction_logs 
              WHERE DATE(log_date) >= '$start_date' AND DATE(log_date) <= '$end_date' 
              ORDER BY log_date DESC";
    
    $result = mysqli_query($con, $query);
    
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    // Write header
    fputcsv($output, ['Log ID', 'Transaction Type', 'Module', 'Related ID', 'User', 'Action', 'Amount', 'Old Value', 'New Value', 'Description', 'IP Address', 'Date']);
    
    // Write data
    while ($row = mysqli_fetch_assoc($result)) {
        fputcsv($output, [
            $row['log_id'],
            $row['transaction_type'],
            $row['related_module'],
            $row['related_id'],
            $row['username'],
            $row['action'],
            $row['amount'],
            $row['old_value'],
            $row['new_value'],
            $row['description'],
            $row['ip_address'],
            $row['log_date']
        ]);
    }
    
    fclose($output);
}

?>
