<?php
/**
 * Database Backup and Restore Utilities
 * Provides functions for backing up and restoring the school database
 */

// Include connection and roles using absolute paths so this file works when requested directly
$root = dirname(__DIR__);
include_once $root . '/includes/conn.php';
include_once $root . '/includes/roles.php';

// Only Super Admin can use these functions
// Normalize session level keys (support legacy and new keys)
$user_level = (int) ($_SESSION['secusyslevel'] ?? $_SESSION['schoolsyslevel'] ?? 0);
if ($user_level !== 1) {
    // Ensure BASE_URL is defined (conn.php defines it). If not, fall back to inferred root
    $login_url = defined('BASE_URL') ? (BASE_URL . '/login.php') : '/login.php';
    header('Location: ' . $login_url);
    exit();
}

class DatabaseBackup {
    private $con;
    private $backup_dir;

    public function __construct($connection) {
        $this->con = $connection;
        // Use project-root/backups directory to keep backups consistent regardless of current working dir
        $this->backup_dir = dirname(__DIR__) . DIRECTORY_SEPARATOR . 'backups' . DIRECTORY_SEPARATOR;

        // Create backups directory if it doesn't exist
        if (!is_dir($this->backup_dir)) {
            mkdir($this->backup_dir, 0755, true);
        }
    }
    
    /**
     * Create a complete database backup
     * @return array Array with status and filename
     */
    public function createBackup() {
        try {
            $database = 'kingschool';
            $timestamp = date('Y-m-d_H-i-s');
            $filename = $this->backup_dir . 'backup_' . $timestamp . '.sql';
            
            // Get all tables
            $tables = [];
            $result = mysqli_query($this->con, "SHOW TABLES");
            while ($row = mysqli_fetch_array($result)) {
                $tables[] = $row[0];
            }
            
            $output = "-- Database Backup\n";
            $output .= "-- Generated: " . date('Y-m-d H:i:s') . "\n";
            $output .= "-- Database: $database\n";
            $output .= "-- Host: " . $_SERVER['HTTP_HOST'] . "\n\n";
            
            foreach ($tables as $table) {
                $output .= $this->backupTable($table);
            }
            
            // Write to file
            if (file_put_contents($filename, $output)) {
                return [
                    'status' => 'success',
                    'message' => 'Backup created successfully',
                    'filename' => $filename,
                    'size' => filesize($filename)
                ];
            } else {
                return [
                    'status' => 'error',
                    'message' => 'Failed to create backup file'
                ];
            }
        } catch (Exception $e) {
            return [
                'status' => 'error',
                'message' => 'Backup error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Backup a single table
     * @param string $table Table name
     * @return string SQL backup for the table
     */
    private function backupTable($table) {
        $sql = "-- Table: $table\n";
        $sql .= "DROP TABLE IF EXISTS `$table`;\n";
        
        // Get CREATE TABLE statement
        $create = mysqli_query($this->con, "SHOW CREATE TABLE `$table`");
        $row = mysqli_fetch_row($create);
        $sql .= $row[1] . ";\n\n";
        
        // Get table data
        $select = mysqli_query($this->con, "SELECT * FROM `$table`");
        while ($row = mysqli_fetch_assoc($select)) {
            $columns = implode('`, `', array_keys($row));
            $values = array_values($row);
            
            foreach ($values as &$value) {
                if ($value === null) {
                    $value = 'NULL';
                } else {
                    $value = "'" . mysqli_real_escape_string($this->con, $value) . "'";
                }
            }
            
            $values_str = implode(', ', $values);
            $sql .= "INSERT INTO `$table` (`$columns`) VALUES ($values_str);\n";
        }
        
        $sql .= "\n";
        return $sql;
    }
    
    /**
     * Restore database from backup file
     * @param string $backup_file Path to backup file
     * @return array Status and message
     */
    public function restoreBackup($backup_file) {
        try {
            if (!file_exists($backup_file)) {
                return [
                    'status' => 'error',
                    'message' => 'Backup file not found'
                ];
            }
            
            $sql = file_get_contents($backup_file);
            
            // Split SQL statements
            $statements = explode(';', $sql);
            
            $executed = 0;
            $failed = 0;
            
            foreach ($statements as $statement) {
                $statement = trim($statement);
                
                if (empty($statement) || strpos($statement, '--') === 0) {
                    continue;
                }
                
                if (mysqli_query($this->con, $statement)) {
                    $executed++;
                } else {
                    $failed++;
                    error_log('SQL Error: ' . mysqli_error($this->con) . ' - Statement: ' . substr($statement, 0, 100));
                }
            }
            
            return [
                'status' => 'success',
                'message' => 'Restore completed',
                'executed' => $executed,
                'failed' => $failed
            ];
        } catch (Exception $e) {
            return [
                'status' => 'error',
                'message' => 'Restore error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get list of available backups
     * @return array List of backup files
     */
    public function getAvailableBackups() {
        $backups = [];
        
        if (is_dir($this->backup_dir)) {
            $files = scandir($this->backup_dir, SCANDIR_SORT_DESCENDING);
            
            foreach ($files as $file) {
                if (strpos($file, 'backup_') === 0 && pathinfo($file, PATHINFO_EXTENSION) === 'sql') {
                    $filepath = $this->backup_dir . $file;
                    $backups[] = [
                        'filename' => $file,
                        'filepath' => $filepath,
                        'size' => filesize($filepath),
                        'created' => filemtime($filepath),
                        'size_formatted' => $this->formatBytes(filesize($filepath))
                    ];
                }
            }
        }
        
        return $backups;
    }
    
    /**
     * Delete a backup file
     * @param string $filename Filename to delete
     * @return bool Success status
     */
    public function deleteBackup($filename) {
        $filepath = $this->backup_dir . basename($filename);
        
        if (file_exists($filepath) && strpos($filename, 'backup_') === 0) {
            return unlink($filepath);
        }
        
        return false;
    }
    
    /**
     * Format bytes to human readable format
     * @param int $bytes Number of bytes
     * @return string Formatted size
     */
    private function formatBytes($bytes, $precision = 2) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
    
    /**
     * Get database statistics
     * @return array Database stats
     */
    public function getDatabaseStats() {
        $stats = [
            'database' => 'kingschool',
            'tables' => 0,
            'total_records' => 0,
            'total_size' => 0,
            'tables_detail' => []
        ];
        
        $result = mysqli_query($this->con, "SHOW TABLES");
        
        while ($row = mysqli_fetch_array($result)) {
            $table = $row[0];
            $stats['tables']++;
            
            // Count records
            $count_result = mysqli_query($this->con, "SELECT COUNT(*) FROM `$table`");
            $count_row = mysqli_fetch_array($count_result);
            $count = $count_row[0];
            $stats['total_records'] += $count;
            
            // Get table size
            $size_result = mysqli_query($this->con, "SELECT (data_length + index_length) as size FROM information_schema.TABLES WHERE table_schema = 'kingschool' AND table_name = '$table'");
            $size_row = mysqli_fetch_array($size_result);
            $size = $size_row['size'] ?? 0;
            $stats['total_size'] += $size;
            
            $stats['tables_detail'][] = [
                'name' => $table,
                'records' => $count,
                'size' => $size,
                'size_formatted' => $this->formatBytes($size)
            ];
        }
        
        return $stats;
    }
}

// Handle backup requests
if (isset($_GET['action'])) {
    $backup = new DatabaseBackup($con);
    
    switch ($_GET['action']) {
        case 'create':
            $result = $backup->createBackup();
            $_SESSION['message'] = $result['message'];
            $_SESSION['msg_type'] = ($result['status'] == 'success') ? 'success' : 'danger';
            header('Location: ' . BASE_URL . '/backupmanagement.php');
            break;
            
        case 'delete':
            if (isset($_GET['file'])) {
                if ($backup->deleteBackup($_GET['file'])) {
                    $_SESSION['message'] = 'Backup file deleted';
                    $_SESSION['msg_type'] = 'success';
                } else {
                    $_SESSION['message'] = 'Failed to delete backup file';
                    $_SESSION['msg_type'] = 'danger';
                }
            }
            header('Location: ' . BASE_URL . '/backupmanagement.php');
            break;
    }
}

?>
