<?php
if (session_status() !== PHP_SESSION_ACTIVE) session_start();
include __DIR__ . '/../includes/conn.php';

// Only Head of Section allowed here (view-only)
$level = (int)($_SESSION['schoolsyslevel'] ?? 0);
if ($level !== 4) {
    header('Location: login.php');
    exit;
}

// Helper function to format dates
function normalize_date_for_display($val)
{
    if (empty($val) && $val !== '0') {
        return '';
    }
    
    // Check if it's a proper DATE format (Y-m-d)
    $d = DateTime::createFromFormat('Y-m-d', $val);
    if ($d && $d->format('Y-m-d') === $val) {
        return $d->format('d/M/Y');
    }
    
    // Check if it's numeric
    if (is_numeric($val)) {
        $ts = (int) $val;
        // If it's just a 4-digit year (like "2024"), return it as-is with a note
        if ($ts >= 1900 && $ts <= 2100 && strlen((string)$val) === 4) {
            return 'Invalid date (' . $val . ')';
        }
        // If it's a proper timestamp (9-10 digits)
        if ($ts > 31536000) { // Greater than 1971-01-01
            return date('d/M/Y', $ts);
        }
        return '';
    }
    
    // Try other common formats
    if (!$d) $d = DateTime::createFromFormat('Y/m/d', $val);
    if (!$d) $d = DateTime::createFromFormat('d-m-Y', $val);
    if (!$d) $d = DateTime::createFromFormat('d/m/Y', $val);
    if ($d) {
        return $d->format('d/M/Y');
    }
    
    // Last resort: try strtotime
    $ts = strtotime($val);
    if ($ts !== false && $ts > 0) {
        return date('d/M/Y', $ts);
    }
    
    return 'Invalid date';
}

$id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
$id = $id ? (int)$id : 0;
if ($id <= 0) {
    echo 'Invalid id';
    exit;
}

$wp_sql = "SELECT w.weeklyplan_id, w.startdate, w.enddate, w.weeklyplandoc, cs.class_id, cs.subject_id, c.class, s.subject, ft.term AS ft_term, ft.year AS ft_year, w.periodsperweek, w.bookpages, w.year AS w_year
    FROM weeklyplans w
    JOIN class_subjects cs ON w.class_sub_id = cs.class_sub_id
    JOIN classes c ON cs.class_id = c.class_id
    JOIN subjects s ON cs.subject_id = s.subject_id
    LEFT JOIN financialterms ft ON ft.financialterm_id = w.term AND ft.status = 1
    WHERE w.weeklyplan_id = ? AND w.status = 1 LIMIT 1";

$wp_stmt = mysqli_prepare($con, $wp_sql);
if ($wp_stmt) {
    mysqli_stmt_bind_param($wp_stmt, 'i', $id);
    mysqli_stmt_execute($wp_stmt);
    $res = mysqli_stmt_get_result($wp_stmt);
    $row = $res ? mysqli_fetch_assoc($res) : null;
    mysqli_stmt_close($wp_stmt);
} else {
    $row = null;
}

if (!$row) {
    echo 'Weekly plan not found';
    exit;
}

// show admin header (sidebar)
include __DIR__ . '/../includes/header.php';
?>
<!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Weekly Plan (Head) | <?php echo htmlspecialchars($row['subject'] . ' - ' . $row['class']); ?></title>
  <link href="<?php echo BASE_URL; ?>/css/style.css" rel="stylesheet">
  <link href="<?php echo BASE_URL; ?>/css/style-responsive.css" rel="stylesheet">
</head>
<body class="sticky-header">
<section>
  <section class="wrapper">
    <div class="row">
      <div class="col-lg-8">
        <section class="panel">
          <header class="panel-heading"><?php echo htmlspecialchars($row['subject'] . ' : ' . $row['class']); ?> — Weekly Plan</header>
          <div class="panel-body">
            <table class="table table-striped">
              <tr><th>Week Start</th><td><?php echo htmlspecialchars(normalize_date_for_display($row['startdate'])); ?></td></tr>
              <tr><th>Week End</th><td><?php echo htmlspecialchars(normalize_date_for_display($row['enddate'])); ?></td></tr>
              <tr><th>Periods per Week</th><td><?php echo htmlspecialchars($row['periodsperweek']); ?></td></tr>
              <tr><th>Term</th><td><?php echo htmlspecialchars(($row['ft_term'] ?? '') !== '' ? 'Term ' . $row['ft_term'] : ''); ?></td></tr>
              <tr><th>Year</th><td><?php echo htmlspecialchars($row['ft_year'] ?? ''); ?></td></tr>
            </table>
          </div>
        </section>
        <?php if (!empty($row['weeklyplandoc'])): ?>
          <?php 
          $pdf_path = $row['weeklyplandoc'];
          // Clean up the path - remove any leading slashes or backslashes
          $pdf_path = ltrim($pdf_path, '/\\');
          // Construct the full URL - the path should be relative to the root
          $iframe_src = BASE_URL . '/' . $pdf_path;
          ?>
          <section class="panel">
            <header class="panel-heading">Weekly Plan Document</header>
            <div class="panel-body">
              <?php
              // Check if file exists - go up one level from 'head' directory to root
              $file_path = dirname(__DIR__) . '/' . $pdf_path;
              if (file_exists($file_path)): ?>
                <div style="height:800px;">
                  <iframe src="<?php echo htmlspecialchars($iframe_src, ENT_QUOTES); ?>" width="100%" height="100%" style="border:1px solid #ddd;" type="application/pdf"></iframe>
                </div>
                <p style="margin-top:10px;"><a href="<?php echo htmlspecialchars($iframe_src, ENT_QUOTES); ?>" target="_blank" class="btn btn-primary">Open in New Tab</a></p>
              <?php else: ?>
                <div class="alert alert-danger">
                  <strong>⚠ Document Not Found</strong><br>
                  The document file is missing from the server. Please contact the teacher to re-upload the document.
                </div>
              <?php endif; ?>
            </div>
          </section>
        <?php else: ?>
          <div class="alert alert-info">No document uploaded for this weekly plan.</div>
        <?php endif; ?>
      </div>
    </div>
  </section>
</section>
</body>
</html>
