<?php
// find_substitute.php
// Accepts POST: date (Y-m-d), starttime (HH:MM), endtime (HH:MM), class_sub_id (optional int)
// Returns JSON list of available teachers. Prepared statements used with deterministic escaped fallback.

session_start();
header('Content-Type: application/json; charset=utf-8');

// Don't fatal if a specific config file is missing; try several common locations for DB connection.
// This keeps the endpoint usable whether the app uses config.php or includes/conn.php.
if (file_exists(__DIR__ . '/config.php')) {
    include_once __DIR__ . '/config.php';
} elseif (file_exists(__DIR__ . '/includes/conn.php')) {
    include_once __DIR__ . '/includes/conn.php';
} elseif (file_exists(__DIR__ . '/db_connect.php')) {
    include_once __DIR__ . '/db_connect.php';
} elseif (file_exists(__DIR__ . '/../includes/conn.php')) {
    include_once __DIR__ . '/../includes/conn.php';
}

// Try to use $con from included config, else try common connection files
$con = $con ?? null;
if (!$con) {
    // try project common conn file
    if (file_exists(__DIR__ . '/includes/conn.php')) {
        include __DIR__ . '/includes/conn.php';
        $con = $con ?? null;
    }
}
if (!$con) {
    if (file_exists(__DIR__ . '/db_connect.php')) {
        include __DIR__ . '/db_connect.php';
        $con = $con ?? null;
    }
}

if (!$con) {
    http_response_code(500);
    echo json_encode(['error' => 'DB connection not available']);
    exit;
}

// Permission: only logged-in teachers or admins may use (adjust as needed)
if (!isset($_SESSION['teachersys']) && !isset($_SESSION['schoolsyslevel'])) {
    http_response_code(403);
    echo json_encode(['error' => 'Forbidden']);
    exit;
}

// Helper: safe prepare + fallback
function db_prepare_or_query($con, $sql, $types = '', $params = []) {
    // try prepared
    $stmt = @mysqli_prepare($con, $sql);
    if ($stmt) {
        if (!empty($types) && !empty($params)) {
            mysqli_stmt_bind_param($stmt, $types, ...$params);
        }
        if (!mysqli_stmt_execute($stmt)) {
            error_log('DB execute error: ' . mysqli_error($con) . ' SQL: ' . $sql);
            return false;
        }
        $res = mysqli_stmt_get_result($stmt);
        mysqli_stmt_close($stmt);
        return $res;
    }
    // fallback: build a safe escaped query string
    foreach ($params as $i => $p) {
        if (is_string($p)) $params[$i] = "'" . mysqli_real_escape_string($con, $p) . "'";
        elseif (is_null($p)) $params[$i] = 'NULL';
        else $params[$i] = (int)$p;
    }
    // Very basic fallback replacement for '?' tokens: replace in order
    $q = '';
    $parts = explode('?', $sql);
    for ($i = 0; $i < count($parts) - 1; $i++) {
        $q .= $parts[$i] . ($params[$i] ?? 'NULL');
    }
    $q .= end($parts);
    $res = mysqli_query($con, $q);
    if (!$res) {
        error_log('DB fallback error: ' . mysqli_error($con) . ' SQL: ' . $q);
        return false;
    }
    return $res;
}

// Helper: normalize incoming date to Y-m-d regardless of input format
function normalize_date_find($s) {
    $s = trim((string)$s);
    if ($s === '') return null;
    // Try d/m/Y FIRST (common in many regions: 06/11/2025 = 6 Nov 2025)
    $fmts = ['d/m/Y','Y-m-d','d-m-Y','m/d/Y','m-d-Y'];
    foreach ($fmts as $fmt) {
        $dt = DateTime::createFromFormat($fmt, $s);
        if ($dt && $dt->format($fmt) === $s) return $dt->format('Y-m-d');
    }
    $ts = strtotime($s);
    if ($ts) return date('Y-m-d', $ts);
    return null;
}

// Read inputs
$date = normalize_date_find($_POST['date'] ?? '');
$start = trim($_POST['starttime'] ?? '');
$end = trim($_POST['endtime'] ?? '');
$class_sub_id = isset($_POST['class_sub_id']) ? (int)$_POST['class_sub_id'] : 0;

// Basic validation
$errors = [];
if (empty($date)) $errors[] = 'Invalid date';
if (empty($start)) $errors[] = 'starttime required';
if (empty($end)) $errors[] = 'endtime required';

if ($errors) {
    http_response_code(400);
    echo json_encode(['error' => implode('; ', $errors)]);
    exit;
}

$day = (new DateTime($date))->format('l'); // Monday, Tuesday, etc.

// 1) find teachers who are busy at that day/time (based on timetables + teacher_subjects)
$busy_sql = "SELECT DISTINCT ts.teacher_id FROM timetables t JOIN teacher_subjects ts ON CAST(t.activity AS UNSIGNED) = ts.class_sub_id WHERE t.day LIKE ? AND t.starttime = ? AND t.endtime = ? AND t.status = 1";
$dayLike = substr($day, 0, 3) . '%';
$busy_res = db_prepare_or_query($con, $busy_sql, 'sss', [$dayLike, $start, $end]);
$busy_ids = [];
if ($busy_res && mysqli_num_rows($busy_res) > 0) {
    while ($r = mysqli_fetch_assoc($busy_res)) $busy_ids[] = (int)$r['teacher_id'];
}

// 2) find teachers marked absent for that date (teacherattendance -> teacherattendancedetails)
$absent_ids = [];
$att_stmt = mysqli_prepare($con, "SELECT teacherattendance_id FROM teacherattendance WHERE date = ? AND status = 1 LIMIT 1");
if ($att_stmt) {
    mysqli_stmt_bind_param($att_stmt, 's', $date);
    mysqli_stmt_execute($att_stmt);
    $ar = mysqli_stmt_get_result($att_stmt);
    if ($ar && mysqli_num_rows($ar) > 0) {
        $att_row = mysqli_fetch_assoc($ar);
        $ta_id = (int)($att_row['teacherattendance_id'] ?? 0);
        if ($ta_id > 0) {
            $ad_stmt = mysqli_prepare($con, "SELECT teacher_id FROM teacherattendancedetails WHERE teacherattendance_id = ? AND status = 0");
            if ($ad_stmt) {
                mysqli_stmt_bind_param($ad_stmt, 'i', $ta_id);
                mysqli_stmt_execute($ad_stmt);
                $adr = mysqli_stmt_get_result($ad_stmt);
                if ($adr) while ($r = mysqli_fetch_assoc($adr)) $absent_ids[] = (int)$r['teacher_id'];
                mysqli_stmt_close($ad_stmt);
            } else {
                // fallback query
                $q = sprintf("SELECT teacher_id FROM teacherattendancedetails WHERE teacherattendance_id = %d AND status = 0", $ta_id);
                $adr = mysqli_query($con, $q);
                if ($adr) while ($r = mysqli_fetch_assoc($adr)) $absent_ids[] = (int)$r['teacher_id'];
            }
        }
    }
    mysqli_stmt_close($att_stmt);
} else {
    // fallback: try direct query
    $q = sprintf("SELECT teacherattendance_id FROM teacherattendance WHERE date = '%s' AND status = 1 LIMIT 1", mysqli_real_escape_string($con, $date));
    $ar = mysqli_query($con, $q);
    if ($ar && mysqli_num_rows($ar) > 0) {
        $att_row = mysqli_fetch_assoc($ar);
        $ta_id = (int)($att_row['teacherattendance_id'] ?? 0);
        if ($ta_id > 0) {
            $q2 = sprintf("SELECT teacher_id FROM teacherattendancedetails WHERE teacherattendance_id = %d AND status = 0", $ta_id);
            $adr = mysqli_query($con, $q2);
            if ($adr) while ($r = mysqli_fetch_assoc($adr)) $absent_ids[] = (int)$r['teacher_id'];
        }
    }
}

// Compose exclusion list
$exclude = array_unique(array_merge($busy_ids, $absent_ids));

// Also exclude teachers already assigned as substitutes for this date/time (active substitutions)
$assigned_ids = [];
$subs_stmt = mysqli_prepare($con, "SELECT substitute_teacher_id FROM substitutions WHERE date = ? AND starttime = ? AND endtime = ? AND status = 1");
if ($subs_stmt) {
    mysqli_stmt_bind_param($subs_stmt, 'sss', $date, $start, $end);
    mysqli_stmt_execute($subs_stmt);
    $sr = mysqli_stmt_get_result($subs_stmt);
    if ($sr) while ($r = mysqli_fetch_assoc($sr)) $assigned_ids[] = (int)$r['substitute_teacher_id'];
    mysqli_stmt_close($subs_stmt);
} else {
    // fallback
    $q = sprintf("SELECT substitute_teacher_id FROM substitutions WHERE date = '%s' AND starttime = '%s' AND endtime = '%s' AND status = 1", mysqli_real_escape_string($con,$date), mysqli_real_escape_string($con,$start), mysqli_real_escape_string($con,$end));
    $sr = mysqli_query($con, $q);
    if ($sr) while ($r = mysqli_fetch_assoc($sr)) $assigned_ids[] = (int)$r['substitute_teacher_id'];
}

if (!empty($assigned_ids)) {
    $exclude = array_unique(array_merge($exclude, $assigned_ids));
}

// 3) fetch available teachers
$params = [];
$types = '';
$sql = "SELECT teacher_id, fullname, designation, email, phone FROM teachers WHERE status = 1";
if (!empty($exclude)) {
    // build placeholders
    $placeholders = implode(',', array_fill(0, count($exclude), '?'));
    // but simpler: we'll build the NOT IN clause using ints safely
    $inlist = implode(',', array_map('intval', $exclude));
    $sql .= " AND teacher_id NOT IN ($inlist)";
}

// We'll run the select
$res = db_prepare_or_query($con, $sql);
if ($res === false) {
    http_response_code(500);
    echo json_encode(['error' => 'DB error while searching teachers']);
    exit;
}

$available = [];
while ($row = mysqli_fetch_assoc($res)) {
    $tid = (int)$row['teacher_id'];
    // determine if teacher teaches the requested subject (if provided)
    $teaches_subject = false;
    if ($class_sub_id > 0) {
        $ts_stmt = mysqli_prepare($con, "SELECT 1 FROM teacher_subjects WHERE teacher_id = ? AND class_sub_id = ? LIMIT 1");
        if ($ts_stmt) {
            mysqli_stmt_bind_param($ts_stmt, 'ii', $tid, $class_sub_id);
            mysqli_stmt_execute($ts_stmt);
            $tr = mysqli_stmt_get_result($ts_stmt);
            $teaches_subject = ($tr && mysqli_num_rows($tr) > 0);
            mysqli_stmt_close($ts_stmt);
        } else {
            $q = sprintf("SELECT 1 FROM teacher_subjects WHERE teacher_id = %d AND class_sub_id = %d LIMIT 1", $tid, $class_sub_id);
            $tr = mysqli_query($con, $q);
            $teaches_subject = ($tr && mysqli_num_rows($tr) > 0);
        }
    }
    $available[] = [
        'teacher_id' => $tid,
        'fullname' => $row['fullname'] ?? '',
        'designation' => $row['designation'] ?? '',
        'email' => $row['email'] ?? '',
        'phone' => $row['phone'] ?? '',
        'teaches_subject' => $teaches_subject,
    ];
}

// Sort: teachers who teach the subject first
usort($available, function ($a, $b) {
    if ($a['teaches_subject'] === $b['teaches_subject']) return strcasecmp($a['fullname'], $b['fullname']);
    return $a['teaches_subject'] ? -1 : 1;
});

echo json_encode(['date' => $date, 'start' => $start, 'end' => $end, 'day' => $day, 'available' => $available]);
exit;

?>
