<?php
include 'includes/conn.php';
include 'includes/roles.php';

if (!canViewFinancialReports($_SESSION['schoolsyslevel'])) {
    header('Location:login.php');
    exit();
}

$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-d');
$start_safe = mysqli_real_escape_string($con, $start_date);
$end_safe = mysqli_real_escape_string($con, $end_date);

// Try different possible payment tables/columns to support variations
// Candidate tables and common column name alternatives
$candidates = [
    'accounts_receivable' => [
        'payer' => ['payer_name','payer','customer_name','name'],
        'amount' => ['amount_paid','amount','paid'],
        'date' => ['updated_at','created_at','date']
    ],
    'feespayments' => [
        'payer' => ['payer_name','payer','name'],
        'amount' => ['amount','amount_paid','paid'],
        'date' => ['created_at','date']
    ],
    'feespayment' => [
        'payer' => ['payer_name','payer','name'],
        'amount' => ['amount','amount_paid','paid'],
        'date' => ['created_at','date']
    ]
];

$payments = [];

function table_exists($con, $table) {
    $t = mysqli_real_escape_string($con, $table);
    $r = @mysqli_query($con, "SHOW TABLES LIKE '".$t."'");
    return $r && mysqli_num_rows($r) > 0;
}

function find_column($con, $db, $table, $candidates) {
    foreach ($candidates as $col) {
        $colEsc = mysqli_real_escape_string($con, $col);
        $q = "SELECT COUNT(*) as cnt FROM information_schema.COLUMNS WHERE TABLE_SCHEMA='".mysqli_real_escape_string($con,$db)."' AND TABLE_NAME='".mysqli_real_escape_string($con,$table)."' AND COLUMN_NAME='".$colEsc."'";
        $r = @mysqli_query($con, $q);
        if ($r) {
            $row = mysqli_fetch_assoc($r);
            if ((int)$row['cnt'] > 0) return $col;
        }
    }
    return false;
}

$dbNameRes = mysqli_query($con, "SELECT DATABASE() as db");
$dbName = $dbNameRes ? mysqli_fetch_assoc($dbNameRes)['db'] : '';

foreach ($candidates as $table => $map) {
    if (!table_exists($con, $table)) continue;
    $payerCol = find_column($con, $dbName, $table, $map['payer']);
    $amountCol = find_column($con, $dbName, $table, $map['amount']);
    $dateCol = find_column($con, $dbName, $table, $map['date']);

    if (!$amountCol || !$dateCol) continue; // need at least amount and date

    // Build select list: alias missing columns to NULL
    $selectParts = [];
    $selectParts[] = $payerCol ? "`".$payerCol."` AS payer" : "NULL AS payer";
    $selectParts[] = "`".$amountCol."` AS amount";

    // Date handling: if column type is int -> FROM_UNIXTIME(column), else use column directly
    $isInt = false;
    $colInfo = @mysqli_query($con, "SHOW COLUMNS FROM `".mysqli_real_escape_string($con,$table)."` LIKE '".mysqli_real_escape_string($con,$dateCol)."'");
    if ($colInfo && ($ci = mysqli_fetch_assoc($colInfo))) {
        $type = strtolower($ci['Type'] ?? '');
        if (strpos($type, 'int') !== false) $isInt = true;
    }
    if ($isInt) {
        $selectParts[] = "FROM_UNIXTIME(`".$dateCol."`) AS created_at";
        $whereDate = "FROM_UNIXTIME(`".$dateCol."`) >= '$start_safe' AND FROM_UNIXTIME(`".$dateCol."`) <= '$end_safe'";
    } else {
        $selectParts[] = "`".$dateCol."` AS created_at";
        $whereDate = "DATE(`".$dateCol."`) >= '$start_safe' AND DATE(`".$dateCol."`) <= '$end_safe'";
    }

    $selectParts[] = "'".mysqli_real_escape_string($con,$table)."' AS source";
    $sql = "SELECT " . implode(', ', $selectParts) . " FROM `".mysqli_real_escape_string($con,$table)."` WHERE " . $whereDate . " LIMIT 1000";

    $r = @mysqli_query($con, $sql);
    if ($r && mysqli_num_rows($r) > 0) {
        while ($row = mysqli_fetch_assoc($r)) {
            $payments[] = $row;
        }
        break; // use first matching table
    }
}

// Total collected
$total = 0.0;
foreach ($payments as $p) {
    $total += (float) ($p['amount'] ?? 0);
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0">
    <link rel="shortcut icon" href="#" type="image/png">
    <title>Fee Collection Report</title>
    <link href="css/style.css" rel="stylesheet">
    <link href="css/style-responsive.css" rel="stylesheet">
</head>
<body class="sticky-header">
<section>
    <?php include 'includes/header.php'; ?>
    <section class="wrapper">
        <div class="row">
            <div class="col-lg-12">
                <section class="panel">
                    <header class="panel-heading">
                        <h4 class="panel-title"><i class="fa fa-money"></i> Fee Collection Report: <?php echo htmlspecialchars($start_date); ?> — <?php echo htmlspecialchars($end_date); ?></h4>
                    </header>
                    <div class="panel-body">
                        <div class="row margin-bottom-20">
                            <div class="col-md-4">
                                <strong>Total Collected:</strong>
                                <h3 class="text-success"><?php echo number_format($total,2); ?></h3>
                            </div>
                            <div class="col-md-8 text-right">
                                <a href="financialreports.php?start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>" class="btn btn-default">Back to Reports</a>
                            </div>
                        </div>
                        <table class="table table-striped table-bordered">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Payer</th>
                                    <th>Amount</th>
                                    <th>Date</th>
                                    <th>Source</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($payments)) { ?>
                                    <tr><td colspan="5">No payments found for the selected period.</td></tr>
                                <?php } else {
                                    $i=1; foreach ($payments as $p) {
                                        $date = $p['created_at'] ?? '';
                                        // Normalize unix timestamp if numeric
                                        if (is_numeric($date)) {
                                            $date = date('Y-m-d H:i:s', (int)$date);
                                        }
                                        echo '<tr>';
                                        echo '<td>'.($i++).'</td>';
                                        echo '<td>'.htmlspecialchars($p['payer'] ?? 'N/A').'</td>';
                                        echo '<td>'.number_format((float)($p['amount'] ?? 0),2).'</td>';
                                        echo '<td>'.htmlspecialchars($date).'</td>';
                                        echo '<td>'.htmlspecialchars($p['source'] ?? '').'</td>';
                                        echo '</tr>';
                                    }
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </section>
            </div>
        </div>
    </section>
</section>
<script src="js/jquery-1.10.2.min.js"></script>
<script src="js/bootstrap.min.js"></script>
<script src="js/scripts.js"></script>
</body>
</html>
