<?php
include 'includes/conn.php';
include 'includes/roles.php';
if (session_status() === PHP_SESSION_NONE) { session_start(); }

// Only users with full financial permissions can delete transactions
if (!hasFullFinancialPermissions($_SESSION['schoolsyslevel'])) {
    header('Location: login.php');
    exit;
}

$id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$id) {
    header('Location: banktransactions.php');
    exit;
}

// Fetch the transaction
$txQ = mysqli_prepare($con, "SELECT * FROM bank_transactions WHERE transaction_id = ? LIMIT 1");
if (!$txQ) { header('Location: banktransactions.php'); exit; }
mysqli_stmt_bind_param($txQ, 'i', $id);
mysqli_stmt_execute($txQ);
$txRes = mysqli_stmt_get_result($txQ);
$tx = mysqli_fetch_assoc($txRes);
mysqli_stmt_close($txQ);

if (!$tx) {
    header('Location: banktransactions.php');
    exit;
}

$bank_account_id = (int)$tx['bank_account_id'];

// Ensure this is the latest transaction for the account to avoid history inconsistency
$latestQ = mysqli_query($con, "SELECT transaction_id FROM bank_transactions WHERE bank_account_id = $bank_account_id ORDER BY transaction_date DESC, transaction_time DESC, transaction_id DESC LIMIT 1");
$latest = mysqli_fetch_assoc($latestQ);
if (!$latest || (int)$latest['transaction_id'] !== $id) {
    $error_message = 'Deletion is only allowed for the most recent transaction for this account. To correct older entries, create a reversing transaction.';
}

// Handle deletion confirmation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm_delete']) && empty($error_message)) {
    // Recompute balance and delete inside DB transaction
    mysqli_begin_transaction($con);
    try {
        // Lock bank account row
        $balStmt = mysqli_prepare($con, "SELECT balance FROM bank_accounts WHERE bank_account_id = ? FOR UPDATE");
        if (!$balStmt) throw new Exception('Prepare failed: '.mysqli_error($con));
        mysqli_stmt_bind_param($balStmt, 'i', $bank_account_id);
        mysqli_stmt_execute($balStmt);
        $balRes = mysqli_stmt_get_result($balStmt);
        $balRow = mysqli_fetch_assoc($balRes);
        mysqli_stmt_close($balStmt);

        if (!$balRow) throw new Exception('Bank account not found');
        $current_balance = (float)$balRow['balance'];

        // Reverse the transaction effect
        $amount = (float)$tx['amount'];
        $type = $tx['transaction_type'];
        if ($type === 'Deposit' || $type === 'Interest') {
            $new_balance = $current_balance - $amount;
        } else {
            $new_balance = $current_balance + $amount;
        }

        // Delete the transaction
        $del = mysqli_prepare($con, "DELETE FROM bank_transactions WHERE transaction_id = ?");
        if (!$del) throw new Exception('Prepare failed: '.mysqli_error($con));
        mysqli_stmt_bind_param($del, 'i', $id);
        if (!mysqli_stmt_execute($del)) throw new Exception('Delete failed: '.mysqli_error($con));
        mysqli_stmt_close($del);

        // Update account balance
        $upd = mysqli_prepare($con, "UPDATE bank_accounts SET balance = ? WHERE bank_account_id = ?");
        if (!$upd) throw new Exception('Prepare failed: '.mysqli_error($con));
        mysqli_stmt_bind_param($upd, 'di', $new_balance, $bank_account_id);
        if (!mysqli_stmt_execute($upd)) throw new Exception('Balance update failed: '.mysqli_error($con));
        mysqli_stmt_close($upd);

        mysqli_commit($con);
        header('Location: banktransactions.php?msg=deleted');
        exit;
    } catch (Exception $e) {
        mysqli_rollback($con);
        $error_message = 'Failed to delete transaction: ' . $e->getMessage();
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Delete Bank Transaction</title>
  <link rel="stylesheet" href="css/style.css">
  <link rel="stylesheet" href="css/style-responsive.css">
</head>
<body class="sticky-header">
<section>
  <?php include 'includes/header.php'; ?>
  <section class="wrapper">
    <div class="row">
      <div class="col-lg-8 col-lg-offset-2">
        <section class="panel">
          <header class="panel-heading bg-danger">
            <i class="fa fa-trash"></i> Delete Bank Transaction
          </header>
          <div class="panel-body">
            <?php if (!empty($error_message)): ?>
              <div class="alert alert-danger"><?php echo htmlspecialchars($error_message); ?></div>
            <?php endif; ?>

            <table class="table table-bordered">
              <tr><th>Date</th><td><?php echo htmlspecialchars($tx['transaction_date']); ?></td></tr>
              <tr><th>Time</th><td><?php echo htmlspecialchars($tx['transaction_time']); ?></td></tr>
              <tr><th>Type</th><td><?php echo htmlspecialchars($tx['transaction_type']); ?></td></tr>
              <tr><th>Amount</th><td><?php echo number_format($tx['amount'],2); ?></td></tr>
              <tr><th>Reference</th><td><?php echo htmlspecialchars($tx['reference_number']); ?></td></tr>
              <tr><th>Description</th><td><?php echo nl2br(htmlspecialchars($tx['description'])); ?></td></tr>
              <tr><th>Balance After</th><td><?php echo number_format($tx['balance_after'] ?? 0,2); ?></td></tr>
            </table>

            <form method="post" action="">
              <p class="text-danger"><strong>Are you sure you want to permanently delete this transaction?</strong></p>
              <button type="submit" name="confirm_delete" value="1" class="btn btn-danger"><i class="fa fa-trash"></i> Yes, Delete</button>
              <a href="transactionsbyaccount.php?id=<?php echo $bank_account_id; ?>" class="btn btn-default">Cancel</a>
            </form>
          </div>
        </section>
      </div>
    </div>
  </section>
</section>
<script src="js/jquery-1.10.2.min.js"></script>
<script src="js/bootstrap.min.js"></script>
</body>
</html>
