<?php
include 'includes/conn.php';
session_start();

// Basic permission check - accept legacy session key names
// The main login sets `$_SESSION['schoolsys']`, older code may use `user_id`.
$session_user = $_SESSION['user_id'] ?? $_SESSION['schoolsys'] ?? null;
if (empty($session_user)) {
    header('Location: login.php');
    exit();
}

$requested_by = (int)$session_user;
$purpose = trim($_POST['purpose'] ?? '');
$total_items = (int)($_POST['total_items'] ?? 0);
$items_raw = trim($_POST['requisition_items'] ?? '');

if ($purpose === '' || $total_items <= 0) {
    $_SESSION['message'] = 'Please provide purpose and total items.';
    $_SESSION['msg_type'] = 'danger';
    header('Location: stockrequisitions.php');
    exit();
}

// Try to parse JSON items if provided, otherwise accept plain-text list
$items = [];
if ($items_raw !== '') {
    $decoded = json_decode($items_raw, true);
    if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
        // User provided valid JSON array of items
        $items = $decoded;
    } else {
        // Accept plain-text items: one per line. Try to parse quantity if present.
        $lines = preg_split('/\r?\n/', $items_raw);
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') continue;

            // Try patterns: "qty x description", "description x qty", "description - qty", "item_id:qty"
            $parsed = ['description' => $line, 'qty' => 1];

            // pattern item_id:qty (e.g. 123:4)
            if (preg_match('/^(\d+)\s*:\s*(\d+)$/', $line, $m)) {
                $parsed = ['item_id' => (int)$m[1], 'qty' => (int)$m[2]];
            }
            // pattern qty x description (e.g. 2 x pens)
            elseif (preg_match('/^(\d+)\s*[xX]\s*(.+)$/', $line, $m)) {
                $parsed = ['description' => trim($m[2]), 'qty' => (int)$m[1]];
            }
            // pattern description x qty (e.g. pens x 2)
            elseif (preg_match('/^(.+)\s*[xX]\s*(\d+)$/', $line, $m)) {
                $parsed = ['description' => trim($m[1]), 'qty' => (int)$m[2]];
            }
            // pattern description - qty (e.g. pens - 2)
            elseif (preg_match('/^(.+)\s*[-]\s*(\d+)$/', $line, $m)) {
                $parsed = ['description' => trim($m[1]), 'qty' => (int)$m[2]];
            }

            $items[] = $parsed;
        }
    }
}

// Generate a simple requisition number
$requisition_number = 'REQ' . time();
$requested_date = date('Y-m-d');

$stmt = mysqli_prepare($con, "INSERT INTO stock_requisitions (requisition_number, requested_by, requested_date, requisition_items, total_items, purpose, approval_status, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, 'Pending', NOW(), NOW())");
if ($stmt) {
    $items_json = json_encode($items);
    mysqli_stmt_bind_param($stmt, 'sissis', $requisition_number, $requested_by, $requested_date, $items_json, $total_items, $purpose);
    // Note: types: s (string), i (int), s (string date), s (items_json), i (int), s (purpose)
    // But mysqli expects types count to match; constructing param types accordingly.
    // Because the second param is int, third is string, fourth is string, fifth int, sixth string -> use 'sissis'
    $ok = mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    if ($ok) {
        $_SESSION['message'] = 'Requisition created successfully.';
        $_SESSION['msg_type'] = 'success';
    } else {
        $_SESSION['message'] = 'Failed to create requisition: ' . mysqli_error($con);
        $_SESSION['msg_type'] = 'danger';
    }
} else {
    $_SESSION['message'] = 'Failed to prepare statement: ' . mysqli_error($con);
    $_SESSION['msg_type'] = 'danger';
}

header('Location: stockrequisitions.php');
exit();
