<?php
session_start();
include 'includes/conn.php';
include 'includes/roles.php';

$user_level = (int) ($_SESSION['secusyslevel'] ?? $_SESSION['schoolsyslevel'] ?? 0);
if (!hasFinancialAccess($user_level) && $user_level !== 2) {
    header('Location: index.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'add') {
        // sanitize inputs
        $route_name = $_POST['route_name'] ?? '';
        $vehicle_id = !empty($_POST['vehicle_id']) ? (int) $_POST['vehicle_id'] : null;
        $route_code = $_POST['route_code'] ?? '';
        $pickup_points = $_POST['pickup_points'] ?? '[]';
        $route_distance = isset($_POST['route_distance']) && $_POST['route_distance'] !== '' ? (float) $_POST['route_distance'] : null;
        $estimated_duration = isset($_POST['estimated_duration']) && $_POST['estimated_duration'] !== '' ? (int) $_POST['estimated_duration'] : null;
        $monthly_fee = isset($_POST['monthly_fee']) && $_POST['monthly_fee'] !== '' ? (float) $_POST['monthly_fee'] : null;
        $status = $_POST['status'] ?? 'Active';
        $notes = $_POST['notes'] ?? '';

        // helper: get table columns and types
        $cols_res = mysqli_query($con, "SHOW COLUMNS FROM bus_routes");
        $tableCols = [];
        while ($col = mysqli_fetch_assoc($cols_res)) {
            $tableCols[$col['Field']] = $col['Type'];
        }

        // build data map only for existing columns (handle schema differences)
        $data = [];
        if (isset($tableCols['route_name'])) $data['route_name'] = $route_name;
        if (isset($tableCols['vehicle_id'])) $data['vehicle_id'] = $vehicle_id;
        if (isset($tableCols['route_code'])) $data['route_code'] = $route_code;
        // fallback: some schemas use route_description
        if (!isset($tableCols['route_code']) && isset($tableCols['route_description'])) $data['route_description'] = $route_code ?: $route_name;
        if (isset($tableCols['pickup_points'])) $data['pickup_points'] = $pickup_points;
        if (isset($tableCols['route_distance'])) $data['route_distance'] = $route_distance;
        // fallback mapping for estimated duration/time
        if (isset($tableCols['estimated_duration'])) $data['estimated_duration'] = $estimated_duration;
        elseif (isset($tableCols['estimated_time'])) $data['estimated_time'] = $estimated_duration;
        if (isset($tableCols['monthly_fee'])) $data['monthly_fee'] = $monthly_fee;
        if (isset($tableCols['notes'])) $data['notes'] = $notes;
        if (isset($tableCols['status'])) {
            // convert 'Active'/'Inactive' to numeric if column is tinyint
            $type = strtolower($tableCols['status']);
            if (strpos($type, 'tinyint') !== false) {
                $data['status'] = ($status === 'Active' || $status === '1' || $status === 1) ? 1 : 0;
            } else {
                $data['status'] = $status;
            }
        }

        // build INSERT statement dynamically
        $cols = [];
        $vals = [];
        foreach ($data as $c => $v) {
            $cols[] = "`" . $c . "`";
            if ($v === null) {
                $vals[] = 'NULL';
            } elseif (is_int($v) || is_float($v)) {
                $vals[] = $v;
            } else {
                $vals[] = "'" . mysqli_real_escape_string($con, $v) . "'";
            }
        }

        if (!empty($cols)) {
            $insert = "INSERT INTO bus_routes (" . implode(',', $cols) . ") VALUES (" . implode(',', $vals) . ")";
            if (mysqli_query($con, $insert)) {
                $_SESSION['message'] = 'Bus route added successfully';
            } else {
                $_SESSION['error'] = 'Error adding route: ' . mysqli_error($con);
            }
        } else {
            $_SESSION['error'] = 'No matching columns found in bus_routes table to insert.';
        }
    } elseif ($_POST['action'] === 'edit') {
        $route_id = (int) $_POST['route_id'];
        // sanitize raw inputs
        $route_name = $_POST['route_name'] ?? '';
        $vehicle_id = !empty($_POST['vehicle_id']) ? (int) $_POST['vehicle_id'] : null;
        $route_code = $_POST['route_code'] ?? '';
        $pickup_points = $_POST['pickup_points'] ?? '[]';
        $route_distance = isset($_POST['route_distance']) && $_POST['route_distance'] !== '' ? (float) $_POST['route_distance'] : null;
        $estimated_duration = isset($_POST['estimated_duration']) && $_POST['estimated_duration'] !== '' ? (int) $_POST['estimated_duration'] : null;
        $monthly_fee = isset($_POST['monthly_fee']) && $_POST['monthly_fee'] !== '' ? (float) $_POST['monthly_fee'] : null;
        $status = $_POST['status'] ?? 'Active';
        $notes = $_POST['notes'] ?? '';

        $cols_res = mysqli_query($con, "SHOW COLUMNS FROM bus_routes");
        $tableCols = [];
        while ($col = mysqli_fetch_assoc($cols_res)) {
            $tableCols[$col['Field']] = $col['Type'];
        }

        $data = [];
        if (isset($tableCols['route_name'])) $data['route_name'] = $route_name;
        if (isset($tableCols['vehicle_id'])) $data['vehicle_id'] = $vehicle_id;
        if (isset($tableCols['route_code'])) $data['route_code'] = $route_code;
        if (!isset($tableCols['route_code']) && isset($tableCols['route_description'])) $data['route_description'] = $route_code ?: $route_name;
        if (isset($tableCols['pickup_points'])) $data['pickup_points'] = $pickup_points;
        if (isset($tableCols['route_distance'])) $data['route_distance'] = $route_distance;
        if (isset($tableCols['estimated_duration'])) $data['estimated_duration'] = $estimated_duration;
        elseif (isset($tableCols['estimated_time'])) $data['estimated_time'] = $estimated_duration;
        if (isset($tableCols['monthly_fee'])) $data['monthly_fee'] = $monthly_fee;
        if (isset($tableCols['notes'])) $data['notes'] = $notes;
        if (isset($tableCols['status'])) {
            $type = strtolower($tableCols['status']);
            if (strpos($type, 'tinyint') !== false) {
                $data['status'] = ($status === 'Active' || $status === '1' || $status === 1) ? 1 : 0;
            } else {
                $data['status'] = $status;
            }
        }

        // build UPDATE statement
        $sets = [];
        foreach ($data as $c => $v) {
            if ($v === null) {
                $sets[] = "`$c`=NULL";
            } elseif (is_int($v) || is_float($v)) {
                $sets[] = "`$c`=" . $v;
            } else {
                $sets[] = "`$c`='" . mysqli_real_escape_string($con, $v) . "'";
            }
        }

        if (!empty($sets)) {
            $update = "UPDATE bus_routes SET " . implode(',', $sets) . " WHERE route_id=$route_id";
            if (mysqli_query($con, $update)) {
                $_SESSION['message'] = 'Bus route updated successfully';
            } else {
                $_SESSION['error'] = 'Error updating route: ' . mysqli_error($con);
            }
        } else {
            $_SESSION['error'] = 'No matching columns found in bus_routes table to update.';
        }
    } elseif ($_POST['action'] === 'delete') {
        $route_id = (int) $_POST['route_id'];
        // detect status column type
        $cols_res = mysqli_query($con, "SHOW COLUMNS FROM bus_routes LIKE 'status'");
        $col = mysqli_fetch_assoc($cols_res);
        $stat_update = "status='Inactive'";
        if ($col) {
            $type = strtolower($col['Type']);
            if (strpos($type, 'tinyint') !== false) {
                $stat_update = "status=0";
            }
        }
        $update = "UPDATE bus_routes SET $stat_update WHERE route_id=$route_id";
        if (mysqli_query($con, $update)) {
            $_SESSION['message'] = 'Bus route deactivated successfully';
        } else {
            $_SESSION['error'] = 'Error deactivating route: ' . mysqli_error($con);
        }
    }
    header('Location: busroutes.php');
    exit;
}

$routes_query = "SELECT br.*, v.vehicle_number, v.vehicle_type 
                 FROM bus_routes br 
                 LEFT JOIN vehicles v ON br.vehicle_id = v.vehicle_id 
                 ORDER BY br.status DESC, br.route_name ASC";
$routes_result = mysqli_query($con, $routes_query);

$vehicles_query = "SELECT vehicle_id, vehicle_number, vehicle_type, capacity FROM vehicles WHERE status='Active' AND vehicle_type IN ('Bus','Van') ORDER BY vehicle_number";
$vehicles_result = mysqli_query($con, $vehicles_query);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0">
    <meta name="description" content="">
    <meta name="author" content="ThemeBucket">
    <link rel="shortcut icon" href="#" type="image/png">
    <title>Bus Routes - School Manager</title>
    <link href="js/advanced-datatable/css/demo_page.css" rel="stylesheet" />
    <link href="js/advanced-datatable/css/demo_table.css" rel="stylesheet" />
    <link rel="stylesheet" href="js/data-tables/DT_bootstrap.css" />
    <link href="css/style.css" rel="stylesheet">
    <link href="css/style-responsive.css" rel="stylesheet">
</head>
<body class="sticky-header">
<section>
    <?php include 'includes/header.php'; ?>
    
    <section class="wrapper">
        <section class="section">
            <div class="row">
                <div class="col-lg-12">
                    <section class="panel">
                        <header class="panel-heading">
                            <h4 class="panel-title">
                                <i class="fa fa-map-marker"></i> Bus Routes Management
                                <span class="pull-right">
                                    <button class="btn btn-sm btn-primary" data-toggle="modal" data-target="#addRouteModal"><i class="fa fa-plus"></i> Add Route</button>
                                </span>
                            </h4>
                        </header>
                        
                        <div class="panel-body">
                            <?php if (isset($_SESSION['message'])): ?>
                                <div class="alert alert-success"><?php echo htmlspecialchars($_SESSION['message']); unset($_SESSION['message']); ?></div>
                            <?php endif; ?>
                            <?php if (isset($_SESSION['error'])): ?>
                                <div class="alert alert-danger"><?php echo htmlspecialchars($_SESSION['error']); unset($_SESSION['error']); ?></div>
                            <?php endif; ?>
                            
                            <table id="routesTable" class="table table-striped table-bordered">
                                <thead>
                                    <tr>
                                        <th>Route Code</th>
                                        <th>Route Name</th>
                                        <th>Assigned Vehicle</th>
                                        <th>Pickup Points</th>
                                        <th>Distance (km)</th>
                                        <th>Duration (min)</th>
                                        <th>Monthly Fee (BIF)</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while ($route = mysqli_fetch_assoc($routes_result)): 
                                        $pickup_points = json_decode($route['pickup_points'] ?? '[]', true) ?? [];
                                        $pickup_count = count($pickup_points);

                                        // normalize display values and guard missing keys
                                        $display_route_code = $route['route_code'] ?? $route['route_description'] ?? '';
                                        $display_route_name = $route['route_name'] ?? '';
                                        $vehicle_number = $route['vehicle_number'] ?? '';
                                        $vehicle_type = $route['vehicle_type'] ?? '';
                                        $display_distance = isset($route['route_distance']) && $route['route_distance'] !== null && $route['route_distance'] !== '' ? (float) $route['route_distance'] : 0.0;
                                        $display_duration = isset($route['estimated_duration']) && $route['estimated_duration'] !== null && $route['estimated_duration'] !== '' ? (int) $route['estimated_duration'] : (isset($route['estimated_time']) ? (int)$route['estimated_time'] : 0);
                                        $display_monthly = isset($route['monthly_fee']) && $route['monthly_fee'] !== null && $route['monthly_fee'] !== '' ? (float) $route['monthly_fee'] : 0.0;
                                        $raw_status = $route['status'] ?? null;
                                        $is_active = ($raw_status === 'Active' || $raw_status === 'active' || $raw_status === 1 || $raw_status === '1');
                                        $status_text = $is_active ? 'Active' : ($raw_status !== null ? (string)$raw_status : 'Inactive');
                                    ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars((string)$display_route_code); ?></strong></td>
                                        <td><?php echo htmlspecialchars((string)$display_route_name); ?></td>
                                        <td>
                                            <?php if (!empty($vehicle_number)): ?>
                                                <span class="label label-info"><?php echo htmlspecialchars((string)$vehicle_number); ?></span>
                                                <small><?php echo htmlspecialchars((string)$vehicle_type); ?></small>
                                            <?php else: ?>
                                                <span class="text-muted">Not Assigned</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge"><?php echo $pickup_count; ?></span> points
                                            <?php if ($pickup_count > 0): ?>
                                                <br><small class="text-muted"><?php echo htmlspecialchars($pickup_points[0]['name'] ?? ''); ?> → <?php echo htmlspecialchars(end($pickup_points)['name'] ?? ''); ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo number_format($display_distance, 1); ?></td>
                                        <td><?php echo $display_duration; ?></td>
                                        <td><?php echo number_format($display_monthly, 2); ?></td>
                                        <td>
                                            <span class="label label-<?php echo $is_active ? 'success' : 'default'; ?>">
                                                <?php echo htmlspecialchars((string)$status_text); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <button class="btn btn-xs btn-info" onclick='editRoute(<?php echo json_encode($route); ?>)'>
                                                <i class="fa fa-edit"></i> Edit
                                            </button>
                                            <?php if ($is_active): ?>
                                            <form method="POST" style="display:inline;" onsubmit="return confirm('Deactivate this route?');">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="route_id" value="<?php echo $route['route_id']; ?>">
                                                <button type="submit" class="btn btn-xs btn-warning"><i class="fa fa-ban"></i> Deactivate</button>
                                            </form>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    </section>
                </div>
            </div>
        </section>
    </section>
    
    <!-- Add Route Modal -->
    <div class="modal fade" id="addRouteModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title">Add Bus Route</h4>
                </div>
                <form method="POST">
                    <input type="hidden" name="action" value="add">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Route Code *</label>
                                    <input type="text" name="route_code" class="form-control" required placeholder="e.g., RT-001">
                                </div>
                                <div class="form-group">
                                    <label>Route Name *</label>
                                    <input type="text" name="route_name" class="form-control" required placeholder="e.g., CBD - Westlands">
                                </div>
                                <div class="form-group">
                                    <label>Assigned Vehicle</label>
                                    <select name="vehicle_id" class="form-control">
                                        <option value="">Not Assigned</option>
                                        <?php mysqli_data_seek($vehicles_result, 0); while ($vehicle = mysqli_fetch_assoc($vehicles_result)): ?>
                                        <option value="<?php echo $vehicle['vehicle_id']; ?>">
                                            <?php echo htmlspecialchars($vehicle['vehicle_number'] . ' (' . $vehicle['vehicle_type'] . ' - Capacity: ' . $vehicle['capacity'] . ')'); ?>
                                        </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                                <div class="form-group">
                                    <label>Route Distance (km)</label>
                                    <input type="number" name="route_distance" class="form-control" step="0.1" min="0">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Estimated Duration (minutes)</label>
                                    <input type="number" name="estimated_duration" class="form-control" min="0">
                                </div>
                                <div class="form-group">
                                    <label>Monthly Fee (BIF)</label>
                                    <input type="number" name="monthly_fee" class="form-control" step="0.01" min="0">
                                </div>
                                <div class="form-group">
                                    <label>Status</label>
                                    <select name="status" class="form-control">
                                        <option value="Active">Active</option>
                                        <option value="Inactive">Inactive</option>
                                    </select>
                                </div>
                                <div class="form-group">
                                    <label>Notes</label>
                                    <textarea name="notes" class="form-control" rows="2"></textarea>
                                </div>
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Pickup Points (JSON format)</label>
                            <textarea name="pickup_points" id="pickup_points_add" class="form-control" rows="4" placeholder='[{"name":"Point 1","time":"07:00"},{"name":"Point 2","time":"07:15"}]'>[]</textarea>
                            <small class="text-muted">Format: [{"name":"Location Name","time":"HH:MM","latitude":0.0,"longitude":0.0}]</small>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                        <button type="submit" class="btn btn-primary"><i class="fa fa-save"></i> Save Route</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Edit Route Modal -->
    <div class="modal fade" id="editRouteModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title">Edit Bus Route</h4>
                </div>
                <form method="POST">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="route_id" id="edit_route_id">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Route Code *</label>
                                    <input type="text" name="route_code" id="edit_route_code" class="form-control" required>
                                </div>
                                <div class="form-group">
                                    <label>Route Name *</label>
                                    <input type="text" name="route_name" id="edit_route_name" class="form-control" required>
                                </div>
                                <div class="form-group">
                                    <label>Assigned Vehicle</label>
                                    <select name="vehicle_id" id="edit_vehicle_id" class="form-control">
                                        <option value="">Not Assigned</option>
                                        <?php mysqli_data_seek($vehicles_result, 0); while ($vehicle = mysqli_fetch_assoc($vehicles_result)): ?>
                                        <option value="<?php echo $vehicle['vehicle_id']; ?>">
                                            <?php echo htmlspecialchars($vehicle['vehicle_number'] . ' (' . $vehicle['vehicle_type'] . ' - Capacity: ' . $vehicle['capacity'] . ')'); ?>
                                        </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                                <div class="form-group">
                                    <label>Route Distance (km)</label>
                                    <input type="number" name="route_distance" id="edit_route_distance" class="form-control" step="0.1" min="0">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Estimated Duration (minutes)</label>
                                    <input type="number" name="estimated_duration" id="edit_estimated_duration" class="form-control" min="0">
                                </div>
                                <div class="form-group">
                                    <label>Monthly Fee (BIF)</label>
                                    <input type="number" name="monthly_fee" id="edit_monthly_fee" class="form-control" step="0.01" min="0">
                                </div>
                                <div class="form-group">
                                    <label>Status</label>
                                    <select name="status" id="edit_status" class="form-control">
                                        <option value="Active">Active</option>
                                        <option value="Inactive">Inactive</option>
                                    </select>
                                </div>
                                <div class="form-group">
                                    <label>Notes</label>
                                    <textarea name="notes" id="edit_notes" class="form-control" rows="2"></textarea>
                                </div>
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Pickup Points (JSON format)</label>
                            <textarea name="pickup_points" id="edit_pickup_points" class="form-control" rows="4"></textarea>
                            <small class="text-muted">Format: [{"name":"Location Name","time":"HH:MM","latitude":0.0,"longitude":0.0}]</small>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                        <button type="submit" class="btn btn-primary"><i class="fa fa-save"></i> Update Route</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
</section>
    <?php include 'includes/footer.php'; ?>
    <!-- Use consistent bundled jQuery to avoid multiple-version conflicts -->
    <script src="js/jquery-1.11.3.min.js"></script>
    <script src="js/bootstrap.min.js"></script>
    <script class="include" type="text/javascript" src="js/jquery.dcjqaccordion.2.7.js"></script>
    <script src="js/jquery.scrollTo.min.js"></script>
    <script src="js/jQuery-slimScroll-1.3.0/jquery.slimscroll.js"></script>
    <script src="js/jquery.nicescroll.js"></script>
    <script type="text/javascript" language="javascript" src="js/advanced-datatable/js/jquery.dataTables.js"></script>
    <script type="text/javascript" src="js/data-tables/DT_bootstrap.js"></script>
    <script>
        $(document).ready(function() {
            console.log('busroutes ready');
            try {
                $('#routesTable').DataTable({
                    order: [[0, 'asc']]
                });
            } catch (err) {
                console.error('DataTable init error:', err);
            }

            // Fallback: ensure Add Route modal opens even if data-toggle doesn't work
            $(document).on('click', '[data-target="#addRouteModal"]', function(e) {
                console.log('Add Route button clicked (data-target handler)');
                try {
                    $('#addRouteModal').modal('show');
                } catch (err) {
                    console.error('Error showing addRoute modal:', err);
                }
            });
        });
        
        function editRoute(route) {
            try {
                console.log('editRoute called', route);
                $('#edit_route_id').val(route.route_id);
                // Some DB schemas use different keys; guard with fallback
                $('#edit_route_code').val(route.route_code || route.route_code);
                $('#edit_route_name').val(route.route_name || route.route_name);
                $('#edit_vehicle_id').val(route.vehicle_id || '');
                $('#edit_route_distance').val(route.route_distance || '');
                $('#edit_estimated_duration').val(route.estimated_duration || route.estimated_time || '');
                $('#edit_monthly_fee').val(route.monthly_fee || route.monthly_fee);
                // status in DB may be numeric (1/0) or string; normalize
                var statusVal = (route.status === 'Active' || route.status === 1 || route.status === '1') ? 'Active' : 'Inactive';
                $('#edit_status').val(statusVal);
                $('#edit_notes').val(route.notes || '');
                $('#edit_pickup_points').val(route.pickup_points || '[]');
                $('#editRouteModal').modal('show');
            } catch (err) {
                console.error('Error in editRoute():', err);
                try { alert('JavaScript error: ' + err.message); } catch(e){}
            }
        }
    </script>
</body>
</html>
