<?php
// assign_substitute.php
// POST: date (Y-m-d), starttime, endtime, class_id, stream_id, class_sub_id (optional), original_teacher_id, substitute_teacher_id
// Records a substitution in `substitutions` table (created if not exists).

session_start();
header('Content-Type: application/json; charset=utf-8');

if (file_exists(__DIR__ . '/config.php')) {
    include_once __DIR__ . '/config.php';
} elseif (file_exists(__DIR__ . '/includes/conn.php')) {
    include_once __DIR__ . '/includes/conn.php';
} elseif (file_exists(__DIR__ . '/db_connect.php')) {
    include_once __DIR__ . '/db_connect.php';
} elseif (file_exists(__DIR__ . '/../includes/conn.php')) {
    include_once __DIR__ . '/../includes/conn.php';
}

// Try to use $con from included config, else try common connection files
$con = $con ?? null;
if (!$con) {
    if (file_exists(__DIR__ . '/includes/conn.php')) {
        include __DIR__ . '/includes/conn.php';
        $con = $con ?? null;
    }
}
if (!$con) {
    if (file_exists(__DIR__ . '/db_connect.php')) { include __DIR__ . '/db_connect.php'; $con = $con ?? null; }
}

if (!$con) { http_response_code(500); echo json_encode(['error' => 'DB connection not available']); exit; }

// Permission: only admins or teachers allowed. Teachers may only assign substitutes for their class (optional check)
if (!isset($_SESSION['teachersys']) && !isset($_SESSION['schoolsyslevel'])) {
    http_response_code(403); echo json_encode(['error' => 'Forbidden']); exit;
}

// Helper - same as in find_substitute
function db_prepare_or_query($con, $sql, $types = '', $params = []) {
    // reset last debug msg
    $GLOBALS['DB_DEBUG_MSG'] = '';
    $stmt = @mysqli_prepare($con, $sql);
    if ($stmt) {
        // bind params using call_user_func_array with references (required by mysqli)
        if (!empty($types) && !empty($params)) {
            // ensure params array has variables (so references work)
            $bind_params = [];
            $bind_params[] = $types;
            for ($i = 0; $i < count($params); $i++) {
                $bind_params[] = &$params[$i];
            }
            // Attempt to bind; log on failure
            if (!call_user_func_array([$stmt, 'bind_param'], $bind_params)) {
                $msg = 'DB bind_param failed: ' . mysqli_error($con) . ' SQL: ' . $sql;
                error_log($msg);
                $GLOBALS['DB_DEBUG_MSG'] = $msg;
                // fall through to fallback below
            }
        }
        if (!mysqli_stmt_execute($stmt)) { $msg = 'DB execute error: '.mysqli_error($con).' SQL: '.$sql; error_log($msg); $GLOBALS['DB_DEBUG_MSG'] = $msg; mysqli_stmt_close($stmt); return false; }
        // For SELECT queries mysqli_stmt_get_result will return a mysqli_result; for INSERT/UPDATE/DELETE it returns false
        $res = mysqli_stmt_get_result($stmt);
        if ($res === false) {
            // No result set (likely INSERT/UPDATE/DELETE). Treat successful execute as success.
            $affected = mysqli_stmt_affected_rows($stmt);
            mysqli_stmt_close($stmt);
            return ($affected !== -1) ? true : false;
        }
        mysqli_stmt_close($stmt);
        return $res;
    }
    foreach ($params as $i => $p) { if (is_string($p)) $params[$i] = "'".mysqli_real_escape_string($con,$p)."'"; elseif (is_null($p)) $params[$i] = 'NULL'; else $params[$i] = (int)$p; }
    $parts = explode('?', $sql); $q = '';
    for ($i=0;$i<count($parts)-1;$i++) { $q .= $parts[$i] . ($params[$i] ?? 'NULL'); }
    $q .= end($parts);
    $res = mysqli_query($con, $q);
    if (!$res) { $msg = 'DB fallback error: '.mysqli_error($con).' SQL: '.$q; error_log($msg); $GLOBALS['DB_DEBUG_MSG'] = $msg; return false; }
    return $res;
}

// Helper: normalize incoming date to Y-m-d regardless of input format
function normalize_date($s) {
    $s = trim((string)$s);
    if ($s === '') return null;
    // Try d/m/Y FIRST (common in many regions: 06/11/2025 = 6 Nov 2025)
    $fmts = ['d/m/Y','Y-m-d','d-m-Y','m/d/Y','m-d-Y'];
    foreach ($fmts as $fmt) {
        $dt = DateTime::createFromFormat($fmt, $s);
        if ($dt && $dt->format($fmt) === $s) {
            return $dt->format('Y-m-d');
        }
    }
    // Fallback to strtotime (use with caution; ambiguous for mm/dd vs dd/mm)
    $ts = strtotime($s);
    if ($ts) return date('Y-m-d', $ts);
    return null;
}

// Read and validate inputs
$date_raw = $_POST['date'] ?? '';
$date = normalize_date($date_raw);
$start = trim($_POST['starttime'] ?? '');
$end = trim($_POST['endtime'] ?? '');
$class_id = isset($_POST['class_id']) ? (int)$_POST['class_id'] : 0;
$stream_id = isset($_POST['stream_id']) ? (int)$_POST['stream_id'] : 0;
$class_sub_id = isset($_POST['class_sub_id']) ? (int)$_POST['class_sub_id'] : 0;
$orig = isset($_POST['original_teacher_id']) ? (int)$_POST['original_teacher_id'] : 0;
$sub = isset($_POST['substitute_teacher_id']) ? (int)$_POST['substitute_teacher_id'] : 0;

$errors = [];
if (empty($date)) $errors[] = 'Invalid date';
if (empty($start)) $errors[] = 'starttime required';
if (empty($end)) $errors[] = 'endtime required';
if ($class_id <= 0) $errors[] = 'class_id required';
if ($sub <= 0) $errors[] = 'substitute_teacher_id required';

if ($errors) { http_response_code(400); echo json_encode(['error' => implode('; ', $errors)]); exit; }

// Ensure substitute is not already marked absent or busy
$check_busy_sql = "SELECT 1 FROM teacher_subjects ts JOIN timetables t ON CAST(t.activity AS UNSIGNED) = ts.class_sub_id WHERE t.day LIKE ? AND t.starttime = ? AND t.endtime = ? AND ts.teacher_id = ? LIMIT 1";
$dayFull = (new DateTime($date))->format('l');
$dayLike = substr($dayFull, 0, 3) . '%'; // Matches 'Tue', 'Tuesday'; 'Thu', 'Thur', 'Thursday'
$busy = db_prepare_or_query($con, $check_busy_sql, 'sssi', [$dayLike, $start, $end, $sub]);
if ($busy && mysqli_num_rows($busy) > 0) { http_response_code(409); echo json_encode(['error' => 'Teacher busy at that time']); exit; }

// Check teacher attendance absence
$abs_q = sprintf("SELECT tad.teacher_id FROM teacherattendance ta JOIN teacherattendancedetails tad ON ta.teacherattendance_id=tad.teacherattendance_id WHERE ta.date='%s' AND tad.teacher_id=%d AND tad.status=0 LIMIT 1", mysqli_real_escape_string($con, $date), $sub);
$abs_r = mysqli_query($con, $abs_q);
if ($abs_r && mysqli_num_rows($abs_r) > 0) { http_response_code(409); echo json_encode(['error' => 'Teacher marked absent that day']); exit; }

// Create substitutions table if not exists
$create_sql = "CREATE TABLE IF NOT EXISTS substitutions (
    substitution_id INT AUTO_INCREMENT PRIMARY KEY,
    date DATE NOT NULL,
    starttime VARCHAR(10) NOT NULL,
    endtime VARCHAR(10) NOT NULL,
    class_id INT NOT NULL,
    stream_id INT NOT NULL,
    class_sub_id INT DEFAULT NULL,
    original_teacher_id INT DEFAULT NULL,
    substitute_teacher_id INT NOT NULL,
    created_by INT DEFAULT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    status TINYINT DEFAULT 1
)
ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
if (!mysqli_query($con, $create_sql)) {
    $msg = 'Failed to create/check substitutions table: ' . mysqli_error($con) . ' SQL: ' . $create_sql;
    error_log($msg);
    $GLOBALS['DB_DEBUG_MSG'] = $msg;
    // allow continuation; insertion will fail and be logged
}

// Insert
$created_by = isset($_SESSION['schoolsyslevel']) ? (int)$_SESSION['schoolsyslevel'] : (int)($_SESSION['teachersys'] ?? 0);
$ins_sql = "INSERT INTO substitutions (date, starttime, endtime, class_id, stream_id, class_sub_id, original_teacher_id, substitute_teacher_id, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
$res = db_prepare_or_query($con, $ins_sql, 'sssiiiiii', [$date, $start, $end, $class_id, $stream_id, $class_sub_id > 0 ? $class_sub_id : null, $orig > 0 ? $orig : null, $sub, $created_by]);
if ($res === false) {
    // Prepare a safe, non-sensitive generic error for regular users
    $response = ['error' => 'Failed to record substitution'];
    // If caller supplied debug=1 and user is admin-like, include last DB error for debugging only
    $is_admin = isset($_SESSION['schoolsyslevel']) && (int)$_SESSION['schoolsyslevel'] > 0;
    $want_debug = isset($_POST['debug']) && $_POST['debug'] == '1';
    if ($want_debug && $is_admin) {
        $response['debug'] = $GLOBALS['DB_DEBUG_MSG'] ?: mysqli_error($con);
    }
    error_log('Insert substitution failed: ' . ($GLOBALS['DB_DEBUG_MSG'] ?: mysqli_error($con)) . ' SQL: ' . $ins_sql);
    http_response_code(500);
    echo json_encode($response);
    exit;
}

// get inserted id
$last_id = mysqli_insert_id($con);

// Create a notification for the substitute teacher so they are informed
// Reuse the notifications + recipients pattern used elsewhere (teacher/reply.php)
// Best-effort: if notification insertion fails, log but still return success for substitution insertion
try {
    $notif_sender = (int)($_SESSION['schoolsys'] ?? $_SESSION['teachersys'] ?? 0);
    $notif_sendertype = isset($_SESSION['schoolsys']) ? 'Staff' : 'Teachers';
    $notif_messagetype = 'single';
    $notif_status = 1;
    $notif_tstamp = date('Y-m-d H:i:s');

    $subject = '';
    $class_name = '';
    if (!empty($class_sub_id) && (int)$class_sub_id > 0) {
        $cs_q = mysqli_query($con, "SELECT subject_id,class_id FROM class_subjects WHERE class_sub_id='" . (int)$class_sub_id . "' LIMIT 1");
        if ($cs_q && $csr = mysqli_fetch_assoc($cs_q)) {
            $subjid = (int)($csr['subject_id'] ?? 0);
            if ($subjid > 0) {
                $s_q = mysqli_query($con, "SELECT subject FROM subjects WHERE subject_id='" . $subjid . "' LIMIT 1");
                if ($s_q && $sr = mysqli_fetch_assoc($s_q)) $subject = $sr['subject'] ?? '';
            }
            $cid = (int)($csr['class_id'] ?? 0);
            if ($cid > 0) {
                $c_q = mysqli_query($con, "SELECT class FROM classes WHERE class_id='" . $cid . "' LIMIT 1");
                if ($c_q && $cr = mysqli_fetch_assoc($c_q)) $class_name = $cr['class'] ?? '';
            }
        }
    }
    if (empty($class_name) && $class_id > 0) {
        $c_q = mysqli_query($con, "SELECT class FROM classes WHERE class_id='" . (int)$class_id . "' LIMIT 1");
        if ($c_q && $cr = mysqli_fetch_assoc($c_q)) $class_name = $cr['class'] ?? '';
    }

    $notif_text = 'You have been assigned as a substitute' . (!empty($subject) ? ' for ' . $subject : '') . ' on ' . $date . ' from ' . $start . ' to ' . $end . (!empty($class_name) ? ' in ' . $class_name : '');

    $stmt = mysqli_prepare($con, "INSERT INTO notifications (notification,sender,sendertype,messagetype,reciever,timestamp,status) VALUES(?,?,?,?,?,?,?)");
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, 'sissisi', $notif_text, $notif_sender, $notif_sendertype, $notif_messagetype, $sub, $notif_tstamp, $notif_status);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
        $notif_id = mysqli_insert_id($con);
        if ($notif_id) {
            $rstmt = mysqli_prepare($con, "INSERT INTO recipients(notification_id,parent_id,status) VALUES(?,?,?)");
            if ($rstmt) {
                $zero = 0;
                mysqli_stmt_bind_param($rstmt, 'iii', $notif_id, $sub, $zero);
                mysqli_stmt_execute($rstmt);
                mysqli_stmt_close($rstmt);
            }
        }
    }
} catch (Exception $e) {
    error_log('Failed to create substitute notification: ' . $e->getMessage());
}

echo json_encode(['success' => true, 'substitution_id' => $last_id, 'date' => $date, 'day' => $dayFull]);
exit;

?>
