import { Link, usePage } from '@inertiajs/react';
import { formatPriceTriple } from '@/utils/priceFormatter';

export default function Offers({ featuredAccommodations = [] }) {
    const { siteinfo } = usePage().props;
    
    // Get dynamic description
    const offersDescription = siteinfo?.offers_description || '';
    // Generate featured items - use featured accommodations if available, otherwise use 2 placeholders
    const getFeaturedItems = () => {
        if (featuredAccommodations && featuredAccommodations.length > 0) {
            // Take up to 2 featured accommodations
            return featuredAccommodations.slice(0, 2).map((acc) => ({
                id: acc.id != null ? String(acc.id) : '',
                name: acc.room_name,
                image: acc.primary_photo ? `/storage/${acc.primary_photo}` : 'https://placehold.co/800x571',
                price: formatPriceTriple(acc.price_fbu, acc.price_usd, acc.price_eur) || 'Price on request',
                description: acc.description || 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts. Separated they live in Bookmarksgrove right at the coast of the Semantics, a large language ocean.',
            }));
        }
        
        // Return 2 placeholder items if no featured accommodations
        return Array.from({ length: 2 }, (_, index) => ({
            id: `placeholder-${index + 1}`,
            name: index === 0 ? 'Family Room' : 'Presidential Room',
            image: 'https://placehold.co/800x571',
            price: 'Price on request',
            description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts. Separated they live in Bookmarksgrove right at the coast of the Semantics, a large language ocean.',
        }));
    };

    const featuredItems = getFeaturedItems();

    return(
        <>
        <section className="section bg-light">

        <div className="container">
        <div className="row justify-content-center text-center mb-5">
            <div className="col-md-7">
            <h2 className="heading" data-aos="fade">Great Offers</h2>
            {offersDescription && (
                <p data-aos="fade">{offersDescription}</p>
            )}
            </div>
        </div>

        {featuredItems.map((item, index) => (
            <div 
                key={item.id} 
                className="site-block-half d-block d-lg-flex bg-white" 
                data-aos="fade" 
                data-aos-delay={(index + 1) * 100}
            >
                <Link 
                    href={item.id && String(item.id).startsWith('placeholder') ? '#' : `/accommodations/${item.id}`}
                    className={`image d-block bg-image-2 ${index === 1 ? 'order-2' : ''}`}
                    style={{backgroundImage: `url('${item.image}')`}}
                ></Link>
                <div className={`text ${index === 1 ? 'order-1' : ''}`}>
                <span className="d-block mb-4">
                    <span className="display-4 text-primary">{item.price}</span> 
                    <span className="text-uppercase letter-spacing-2"> / per night</span> 
                </span>
                <h2 className="mb-4">{item.name}</h2>
                <p>{item.description}</p>
                <p>
                    <Link 
                        href={item.id && String(item.id).startsWith('placeholder') ? '#' : `/accommodations/${item.id}`}
                        className="btn btn-primary text-white"
                    >
                        Book Now
                    </Link>
                </p>
                </div>
            </div>
        ))}

        </div>
        </section>
        </>
    );
}