import React from 'react';
import { usePage } from '@inertiajs/react';
import { formatPriceTriple } from '@/utils/priceFormatter';

export default function Menu({ foodItems = [], drinkItems = [], dessertItems = [] }){
    const { siteinfo, sectionimages } = usePage().props;

    // Get dynamic background images for each tab
    const foodBg = sectionimages?.menu_background ? `/storage/${sectionimages.menu_background}` : '/assets/user/images/hero_3.jpg';
    const drinksBg = sectionimages?.menu_background_drinks ? `/storage/${sectionimages.menu_background_drinks}` : foodBg;
    const dessertsBg = sectionimages?.menu_background_desserts ? `/storage/${sectionimages.menu_background_desserts}` : foodBg;

    // Track active tab
    const [activeTab, setActiveTab] = React.useState('food');

    const getTabBg = () => {
        if (activeTab === 'drinks') return drinksBg;
        if (activeTab === 'desserts') return dessertsBg;
        return foodBg;
    };
    
    // Get dynamic title and description
    const menuTitle = siteinfo?.menu_title || 'Our Restaurant Menu';
    const menuDescription = siteinfo?.menu_description || '';
    // Fallback items - four per category
    const fallbackItems = {
        Food: [
            {
                id: 1,
                name: 'Sample Food Item 1',
                price_fbu: 0.2,
                price_usd: 20,
                price_eur: 18,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            },
            {
                id: 2,
                name: 'Sample Food Item 2',
                price_fbu: 0.2,
                price_usd: 25,
                price_eur: 22,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            },
            {
                id: 3,
                name: 'Sample Food Item 3',
                price_fbu: 0.2,
                price_usd: 18,
                price_eur: 16,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            },
            {
                id: 4,
                name: 'Sample Food Item 4',
                price_fbu: 0.2,
                price_usd: 22,
                price_eur: 20,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            }
        ],
        Drinks: [
            {
                id: 5,
                name: 'Sample Drink 1',
                price_fbu: 0.2,
                price_usd: 5,
                price_eur: 4.5,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            },
            {
                id: 6,
                name: 'Sample Drink 2',
                price_fbu: 0.2,
                price_usd: 6,
                price_eur: 5.5,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            },
            {
                id: 7,
                name: 'Sample Drink 3',
                price_fbu: 0.2,
                price_usd: 4.5,
                price_eur: 4,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            },
            {
                id: 8,
                name: 'Sample Drink 4',
                price_fbu: 0.2,
                price_usd: 5.5,
                price_eur: 5,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            }
        ],
        Desserts: [
            {
                id: 9,
                name: 'Sample Dessert 1',
                price_fbu: 0.2,
                price_usd: 8,
                price_eur: 7,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            },
            {
                id: 10,
                name: 'Sample Dessert 2',
                price_fbu: 0.2,
                price_usd: 10,
                price_eur: 9,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            },
            {
                id: 11,
                name: 'Sample Dessert 3',
                price_fbu: 0.2,
                price_usd: 9,
                price_eur: 8,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            },
            {
                id: 12,
                name: 'Sample Dessert 4',
                price_fbu: 0.2,
                price_usd: 7.5,
                price_eur: 6.5,
                description: 'Far far away, behind the word mountains, far from the countries Vokalia and Consonantia, there live the blind texts.'
            }
        ]
    };

    // Use database items if available, otherwise use fallback
    const displayFood = foodItems.length > 0 ? foodItems : fallbackItems.Food;
    const displayDrinks = drinkItems.length > 0 ? drinkItems : fallbackItems.Drinks;
    const displayDesserts = dessertItems.length > 0 ? dessertItems : fallbackItems.Desserts;

    // Helper function to render menu items
    const renderMenuItems = (items) => {
        // Split items into two columns
        const midPoint = Math.ceil(items.length / 2);
        const leftColumn = items.slice(0, midPoint);
        const rightColumn = items.slice(midPoint);

        const formatDisplayPrice = (item) => {
            const formatted = formatPriceTriple(item.price_fbu, item.price_usd, item.price_eur);
            if (formatted && formatted.length > 0) return formatted;
            // Fallback to legacy single price if present
            if (item.price) return `${item.price}`;
            return 'Price on request';
        };

        return (
            <div className="row">
                <div className="col-md-6">
                    {leftColumn.map((item) => (
                        <div key={item.id} className="food-menu mb-5">
                            <span className="d-block text-primary h4 mb-3">{formatDisplayPrice(item)}</span>
                            <h3 className="text-white">
                                <a href="#" className="text-white">{item.name}</a>
                            </h3>
                            {item.description && (
                                <p className="text-white text-opacity-7">{item.description}</p>
                            )}
                        </div>
                    ))}
                </div>
                <div className="col-md-6">
                    {rightColumn.map((item) => (
                        <div key={item.id} className="food-menu mb-5">
                            <span className="d-block text-primary h4 mb-3">{formatDisplayPrice(item)}</span>
                            <h3 className="text-white">
                                <a href="#" className="text-white">{item.name}</a>
                            </h3>
                            {item.description && (
                                <p className="text-white text-opacity-7">{item.description}</p>
                            )}
                        </div>
                    ))}
                </div>
            </div>
        );
    };

    return(
        <>
        <section className="section bg-image overlay" style={{backgroundImage: `url('${getTabBg()}')`}}>
        <div className="container">
            <div className="row justify-content-center text-center mb-5">
            <div className="col-md-7">
                <h2 className="heading text-white" data-aos="fade">{menuTitle}</h2>
                {menuDescription && (
                    <p className="text-white" data-aos="fade" data-aos-delay="100">{menuDescription}</p>
                )}
            </div>
            </div>
            <div className="food-menu-tabs" data-aos="fade">
            <ul className="nav nav-tabs mb-5" id="myTab" role="tablist">
                <li className="nav-item">
                <a className={`nav-link letter-spacing-2${activeTab === 'food' ? ' active' : ''}`} id="food-tab" data-toggle="tab" href="#food" role="tab" aria-controls="food" aria-selected={activeTab === 'food'} onClick={() => setActiveTab('food')}>Food</a>
                </li>
                <li className="nav-item">
                <a className={`nav-link letter-spacing-2${activeTab === 'drinks' ? ' active' : ''}`} id="drinks-tab" data-toggle="tab" href="#drinks" role="tab" aria-controls="drinks" aria-selected={activeTab === 'drinks'} onClick={() => setActiveTab('drinks')}>Drinks</a>
                </li>
                <li className="nav-item">
                <a className={`nav-link letter-spacing-2${activeTab === 'desserts' ? ' active' : ''}`} id="desserts-tab" data-toggle="tab" href="#desserts" role="tab" aria-controls="desserts" aria-selected={activeTab === 'desserts'} onClick={() => setActiveTab('desserts')}>Desserts</a>
                </li>
            </ul>
            <div className="tab-content py-5" id="myTabContent">
                <div className={`tab-pane fade text-left${activeTab === 'food' ? ' show active' : ''}`} id="food" role="tabpanel" aria-labelledby="food-tab">
                    {renderMenuItems(displayFood)}
                </div>
                <div className={`tab-pane fade text-left${activeTab === 'drinks' ? ' show active' : ''}`} id="drinks" role="tabpanel" aria-labelledby="drinks-tab">
                    {renderMenuItems(displayDrinks)}
                </div>
                <div className={`tab-pane fade text-left${activeTab === 'desserts' ? ' show active' : ''}`} id="desserts" role="tabpanel" aria-labelledby="desserts-tab">
                    {renderMenuItems(displayDesserts)}
                </div>
            </div>
            </div>
        </div>
        </section>
        </>
    );
}