import { useState, useEffect } from 'react';
import { Link } from '@inertiajs/react';
import { formatPriceTriple } from '@/utils/priceFormatter';

export default function Availability() {
    const [checkIn, setCheckIn] = useState('');
    const [checkOut, setCheckOut] = useState('');
    const [adults, setAdults] = useState(1);
    const [children, setChildren] = useState(0);
    const [loading, setLoading] = useState(false);
    const [results, setResults] = useState([]);
    const [error, setError] = useState('');
    const [showResults, setShowResults] = useState(false);

    useEffect(() => {
        const initDatePickers = () => {
            if (typeof window !== 'undefined' && window.$ && window.$.fn.datepicker) {
                const today = new Date();
                today.setHours(0, 0, 0, 0);

                window.$('#availability_checkin_date').datepicker({
                    format: 'd MM, yyyy',
                    autoclose: true,
                    startDate: today,
                }).on('changeDate', function(e) {
                    if (e.date) {
                        const dateStr = e.date.toISOString().split('T')[0];
                        setCheckIn(dateStr);
                        setShowResults(false);
                        setError('');
                    }
                });

                window.$('#availability_checkout_date').datepicker({
                    format: 'd MM, yyyy',
                    autoclose: true,
                    startDate: today,
                }).on('changeDate', function(e) {
                    if (e.date) {
                        const dateStr = e.date.toISOString().split('T')[0];
                        setCheckOut(dateStr);
                        setShowResults(false);
                        setError('');
                    }
                });

                return true;
            }
            return false;
        };

        if (!initDatePickers()) {
            const checkInterval = setInterval(() => {
                if (initDatePickers()) {
                    clearInterval(checkInterval);
                }
            }, 100);
            return () => clearInterval(checkInterval);
        }
    }, []);

    const handleCheckAvailability = async (e) => {
        e.preventDefault();
        setError('');
        setShowResults(false);

        // Validation
        if (!checkIn || !checkOut) {
            setError('Please select both check-in and check-out dates');
            return;
        }

        const checkInDate = new Date(checkIn);
        const checkOutDate = new Date(checkOut);
        const today = new Date();
        today.setHours(0, 0, 0, 0);

        if (checkInDate < today) {
            setError('Check-in date cannot be in the past');
            return;
        }

        if (checkOutDate <= checkInDate) {
            setError('Check-out date must be after check-in date');
            return;
        }

        setLoading(true);

        try {
            const response = await fetch(`/api/accommodations/available?check_in=${checkIn}&check_out=${checkOut}`);
            const data = await response.json();
            
            if (!response.ok) {
                throw new Error(data.error || 'Failed to check availability');
            }

            setResults(data.accommodations || []);
            setShowResults(true);
            setError('');

            if (data.accommodations && data.accommodations.length === 0) {
                setError('No rooms available for the selected dates. Please try different dates.');
            }
        } catch (err) {
            setError(err.message || 'An error occurred while checking availability. Please try again.');
            setResults([]);
            setShowResults(false);
        } finally {
            setLoading(false);
        }
    };

    return (
        <>
            <section className="section bg-light pb-0">
                <div className="container">
                    <div className="row check-availabilty" id="next">
                        <div className="block-32" data-aos="fade-up" data-aos-offset="-200">
                            <form onSubmit={handleCheckAvailability}>
                                <div className="row">
                                    <div className="col-md-6 mb-3 mb-lg-0 col-lg-3">
                                        <label htmlFor="availability_checkin_date" className="font-weight-bold text-black">Check In</label>
                                        <div className="field-icon-wrap">
                                            <div className="icon"><span className="icon-calendar"></span></div>
                                            <input 
                                                type="text" 
                                                id="availability_checkin_date" 
                                                className="form-control"
                                                required
                                            />
                                        </div>
                                    </div>
                                    <div className="col-md-6 mb-3 mb-lg-0 col-lg-3">
                                        <label htmlFor="availability_checkout_date" className="font-weight-bold text-black">Check Out</label>
                                        <div className="field-icon-wrap">
                                            <div className="icon"><span className="icon-calendar"></span></div>
                                            <input 
                                                type="text" 
                                                id="availability_checkout_date" 
                                                className="form-control"
                                                required
                                            />
                                        </div>
                                    </div>
                                    <div className="col-md-6 mb-3 mb-md-0 col-lg-3">
                                        <div className="row">
                                            <div className="col-md-6 mb-3 mb-md-0">
                                                <label htmlFor="availability_adults" className="font-weight-bold text-black">Adults</label>
                                                <div className="field-icon-wrap">
                                                    <div className="icon"><span className="ion-ios-arrow-down"></span></div>
                                                    <select 
                                                        id="availability_adults" 
                                                        className="form-control"
                                                        value={adults}
                                                        onChange={(e) => setAdults(parseInt(e.target.value))}
                                                    >
                                                        <option value="1">1</option>
                                                        <option value="2">2</option>
                                                        <option value="3">3</option>
                                                        <option value="4">4</option>
                                                        <option value="5">5+</option>
                                                    </select>
                                                </div>
                                            </div>
                                            <div className="col-md-6 mb-3 mb-md-0">
                                                <label htmlFor="availability_children" className="font-weight-bold text-black">Children</label>
                                                <div className="field-icon-wrap">
                                                    <div className="icon"><span className="ion-ios-arrow-down"></span></div>
                                                    <select 
                                                        id="availability_children" 
                                                        className="form-control"
                                                        value={children}
                                                        onChange={(e) => setChildren(parseInt(e.target.value))}
                                                    >
                                                        <option value="0">0</option>
                                                        <option value="1">1</option>
                                                        <option value="2">2</option>
                                                        <option value="3">3</option>
                                                        <option value="4">4+</option>
                                                    </select>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div className="col-md-6 col-lg-3 align-self-end">
                                        <button 
                                            type="submit"
                                            className="btn btn-primary btn-block text-white"
                                            disabled={loading}
                                        >
                                            {loading ? 'Checking...' : 'Check Availability'}
                                        </button>
                                    </div>
                                </div>
                                {error && (
                                    <div className="row mt-3">
                                        <div className="col-12">
                                            <div className="alert alert-danger" role="alert">
                                                {error}
                                            </div>
                                        </div>
                                    </div>
                                )}
                            </form>
                        </div>
                    </div>

                    {/* Results Section */}
                    {showResults && (
                        <div className="row mt-5" data-aos="fade-up">
                            <div className="col-12">
                                <h3 className="heading mb-4">Available Room Types</h3>
                                {results.length > 0 ? (
                                    <div className="row">
                                        {results.map((accommodation) => (
                                            <div key={accommodation.id} className="col-md-6 col-lg-4 mb-4">
                                                <Link 
                                                    href={`/accommodations/${accommodation.id}`}
                                                    className="room card h-100 shadow-sm"
                                                    style={{ textDecoration: 'none', color: 'inherit', transition: 'transform 0.2s ease' }}
                                                    onMouseEnter={(e) => e.currentTarget.style.transform = 'translateY(-5px)'}
                                                    onMouseLeave={(e) => e.currentTarget.style.transform = 'translateY(0)'}
                                                >
                                                    <figure className="img-wrap mb-0" style={{ height: '200px', overflow: 'hidden' }}>
                                                        <img 
                                                            src={accommodation.primary_photo ? `/storage/${accommodation.primary_photo}` : 'https://placehold.co/800x571'} 
                                                            alt={accommodation.room_name}
                                                            className="img-fluid"
                                                            style={{ width: '100%', height: '100%', objectFit: 'cover' }}
                                                        />
                                                    </figure>
                                                    <div className="card-body">
                                                        <h4 className="card-title mb-2">
                                                            {accommodation.room_category || accommodation.room_name}
                                                            <span className="badge badge-primary ml-2">({accommodation.available_rooms_count})</span>
                                                        </h4>
                                                        <p className="mb-2" style={{ color: '#c9a227', fontWeight: 'bold' }}>
                                                            {formatPriceTriple(accommodation.price_fbu, accommodation.price_usd, accommodation.price_eur)} / night
                                                        </p>
                                                        <span className="btn btn-sm btn-primary">View Details & Book</span>
                                                    </div>
                                                </Link>
                                            </div>
                                        ))}
                                    </div>
                                ) : (
                                    <div className="alert alert-info" role="alert">
                                        No rooms available for the selected dates. Please try different dates.
                                    </div>
                                )}
                            </div>
                        </div>
                    )}
                </div>
            </section>
        </>
    );
}