import Col from "@admin/ui/Col";
import Row from "@admin/ui/Row";
import StyledTable from "@admin/ui/StyledTable";
import { Link } from "@inertiajs/react";
import DeleteConfirmationModal from "@admin/ui/DeleteConfirmationModal";
import { toast, Toaster } from "react-hot-toast";
import { useState } from "react";
import EditService from "./EditService";

export default function Service({ services }) {
    const columns = [
        { header: '', key: 'checkbox' },
        { header: 'Name', key: 'name' },
        { header: 'Photo', key: 'photo' },
        { header: 'Order', key: 'display_order' },
        { header: 'Status', key: 'status' },
    ];

    const [isEditModalOpen, setIsEditModalOpen] = useState(false);
    const [initialData, setInitialData] = useState(null);
    const [deleteModal, setDeleteModal] = useState({ show: false, service: null });

    const handleDelete = (row) => {
        setDeleteModal({ show: true, service: row });
    };

    const handleEdit = (row) => {
        fetch(`/admin/edit_service/${row.id}`)
            .then((response) => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then((serviceData) => {
                const formattedData = serviceData;
                setInitialData(formattedData);
                setIsEditModalOpen(true);
            })
            .catch((error) => {
                console.error('Error fetching service details:', error);
                toast.error('Failed to load service details');
            });
    };

    const transformedServices = services.map(service => ({
        ...service,
        photo: service.photo ? (
            <img 
                src={`/storage/${service.photo}`} 
                alt={service.name}
                style={{ width: '50px', height: '50px', objectFit: 'cover', borderRadius: '4px' }}
            />
        ) : (
            <span className="text-muted">No image</span>
        ),
        status: service.status ? (
            <span className="badge badge-success">Active</span>
        ) : (
            <span className="badge badge-secondary">Inactive</span>
        ),
    }));

    return (
        <>
            <Toaster />
            <Row md="12">
                <Col lg="12">
                    <div className="d-flex flex-column flex-md-row justify-content-between align-items-start align-items-md-center mb-4">
                        <div className="mb-3 mb-md-0">
                            <h2 className="page-title mb-1">Services</h2>
                            <p className="text-muted mb-0">Manage hotel services displayed on the frontend</p>
                        </div>
                        <Link className="btn btn-primary" href="/admin/services/create">
                            <i className="fe fe-plus fe-16 mr-2"></i>
                            Add New Service
                        </Link>
                    </div>

                    <Row>
                        <Col lg="12">
                            <StyledTable
                                data={transformedServices}
                                columns={columns}
                                itemsPerPage={10}
                                title="Available Services"
                                onEdit={handleEdit}
                                onDelete={handleDelete}
                                noDataText="No services found"
                            />
                        </Col>
                    </Row>
                </Col>
            </Row>

            <DeleteConfirmationModal
                isOpen={deleteModal.show}
                onClose={() => setDeleteModal({ show: false, service: null })}
                onConfirmDelete={() => {
                    toast.success("Service Deleted Successfully");
                }}
                itemName={deleteModal.service?.name || 'this service'}
                itemType="service"
                deleteEndpoint={`/admin/api/delete_service/${deleteModal?.service?.id}`}
            />

            <EditService
                isOpen={isEditModalOpen}
                onClose={() => setIsEditModalOpen(false)}
                initialData={initialData}
            />
        </>
    );
}



