import FormGroup from "@admin/ui/forms/FormGroup";
import Switch from "@admin/ui/forms/Switch";
import FileUploadInput from "@admin/ui/forms/FileUploadInput";
import Modal from "@admin/ui/Modal";
import { useEffect } from "react";
import { useFormResponse } from '@/hooks/useFormResponse';
import { useForm } from "@inertiajs/react";

export default function EditService({ isOpen, onClose, initialData }) {
    const { data, setData, put, processing, errors } = useForm({
        name: '',
        description: '',
        photo: '',
        icon: '',
        link: '',
        display_order: 0,
        status: true,
    });

    const handleSuccess = (filename) => {
        setData('photo', filename);
    };

    const handleError = (error) => {
        console.error("Upload error:", error);
    };

    useEffect(() => {
        if (initialData) {
            setData({
                name: initialData?.name || '',
                description: initialData?.description || '',
                photo: initialData?.photo || '',
                icon: initialData?.icon || '',
                link: initialData?.link || '',
                display_order: initialData?.display_order || 0,
                status: initialData?.status === 1 || initialData?.status === true,
            });
        }
    }, [initialData, setData]);

    const handleSave = (event) => {
        event.preventDefault();
        const formResponse = useFormResponse(() => {
            onClose?.();
        });
        put(`/admin/api/update_service/${initialData.id}`, {
            preserveScroll: true,
            onSuccess: formResponse.onSuccess,
            onError: formResponse.onError,
        });
    };

    return (
        <Modal
            isOpen={isOpen}
            onClose={onClose}
            title="Edit Service"
            size="large"
            primaryButtonText="Save Changes"
            onPrimaryAction={handleSave}
            processing={processing}
        >
            <form onSubmit={handleSave}>
                <FormGroup
                    label="Service Name"
                    value={data.name}
                    onChange={(e) => setData('name', e.target.value)}
                    invalidFeedback={errors.name}
                    required
                />

                <FormGroup
                    id="description"
                    label="Description"
                    as="textarea"
                    value={data.description}
                    onChange={(e) => setData('description', e.target.value)}
                    rows="4"
                    invalidFeedback={errors.description}
                />

                <FormGroup
                    id="icon"
                    label="Icon (Optional)"
                    type="text"
                    value={data.icon}
                    onChange={(e) => setData('icon', e.target.value)}
                    invalidFeedback={errors.icon}
                    helpText="Icon class name (e.g., fe fe-activity)"
                />

                <FormGroup
                    id="link"
                    label="Link (Optional)"
                    type="text"
                    value={data.link}
                    onChange={(e) => setData('link', e.target.value)}
                    invalidFeedback={errors.link}
                    helpText="Link to related page (e.g., /rooms, /dining)"
                />

                <FormGroup
                    id="display_order"
                    label="Display Order"
                    type="number"
                    value={data.display_order}
                    onChange={(e) => setData('display_order', parseInt(e.target.value) || 0)}
                    invalidFeedback={errors.display_order}
                />

                <br />
                <FileUploadInput
                    label="Upload Service Photo (800×600px)"
                    buttonText="Choose Photo"
                    inputId="servicePhotoUploadEdit"
                    uploadUrl="/admin/api/service/upload_image"
                    onSuccess={handleSuccess}
                    onError={handleError}
                    previewUrl={data.photo ? `/storage/${data.photo}` : null}
                    helpText="Recommended size: 800×600px (4:3 aspect ratio). Images will be displayed at 250px height with responsive width."
                />

                <Switch
                    id="statusSwitch"
                    label="Active"
                    checked={data.status}
                    onChange={({ target: { checked } }) =>
                        setData(prevData => ({ ...prevData, status: checked }))
                    }
                />
                <br />
            </form>
        </Modal>
    );
}

