import FormGroup from "@admin/ui/forms/FormGroup";
import Switch from "@admin/ui/forms/Switch";
import Modal from "@admin/ui/Modal";
import Row from "@admin/ui/Row";
import Col from "@admin/ui/Col";
import { useEffect } from "react";
import { useFormResponse } from '@/hooks/useFormResponse';
import { useForm } from "@inertiajs/react";

export default function CreateEditMenuItem({ isOpen, onClose, initialData, defaultCategory = 'Food' }) {
    const { data, setData, post, put, processing, errors } = useForm({
        'name': '',
        'category': defaultCategory,
        'price_fbu': '',
        'price_usd': '',
        'price_eur': '',
        'description': '',
        'display_order': '',
        'status': true,
    });

    useEffect(() => {
        if (initialData) {
            setData({
                name: initialData?.name || '',
                category: initialData?.category || defaultCategory,
                price_fbu: initialData?.price_fbu || '',
                price_usd: initialData?.price_usd || '',
                price_eur: initialData?.price_eur || '',
                description: initialData?.description || '',
                display_order: initialData?.display_order || '',
                status: initialData?.status === 1 || initialData?.status === true,
            });
        } else {
            // Reset form for new item
            setData({
                name: '',
                category: defaultCategory,
                price_fbu: '',
                price_usd: '',
                price_eur: '',
                description: '',
                display_order: '',
                status: true,
            });
        }
    }, [initialData, defaultCategory, setData]);

    const handleSave = (event) => {
        event.preventDefault();
        const formResponse = useFormResponse(() => {
            onClose?.();
        });

        if (initialData) {
            // Update existing item
            put(`/admin/api/update_menu_item/${initialData.id}`, {
                preserveScroll: true,
                onSuccess: formResponse.onSuccess,
                onError: formResponse.onError,
            });
        } else {
            // Create new item
            post('/admin/api/create_menu_item', {
                preserveScroll: true,
                onSuccess: formResponse.onSuccess,
                onError: formResponse.onError,
            });
        }
    };

    return (
        <>
            <Modal
                isOpen={isOpen}
                onClose={onClose}
                title={initialData ? `Edit ${initialData.name}` : 'Create New Menu Item'}
                size="large"
                primaryButtonText="Save Changes"
                onPrimaryAction={handleSave}
                processing={processing}
            >
                <form onSubmit={handleSave}>
                    <Row>
                        <Col md="6">
                            <FormGroup
                                id="name"
                                label="Item Name"
                                type="text"
                                value={data.name}
                                onChange={(e) => setData("name", e.target.value)}
                                required
                                invalidFeedback={errors.name}
                            />

                            <FormGroup
                                id="category"
                                label="Category"
                                as="select"
                                value={data.category}
                                onChange={(e) => setData("category", e.target.value)}
                                required
                                invalidFeedback={errors.category}
                            >
                                <option value="Food">Food</option>
                                <option value="Drinks">Drinks</option>
                                <option value="Desserts">Desserts</option>
                            </FormGroup>

                            <FormGroup
                                id="price_fbu"
                                label="Price (BIF)"
                                type="number"
                                value={data.price_fbu}
                                onChange={(e) => setData("price_fbu", e.target.value)}
                                helpText="Primary currency displayed first"
                                invalidFeedback={errors.price_fbu}
                                min="0.01"
                                step="0.01"
                            />

                            <FormGroup
                                id="price_usd"
                                label="Price (USD)"
                                type="number"
                                value={data.price_usd}
                                onChange={(e) => setData("price_usd", e.target.value)}
                                invalidFeedback={errors.price_usd}
                                min="0.01"
                                step="0.01"
                            />

                            <FormGroup
                                id="price_eur"
                                label="Price (EUR)"
                                type="number"
                                value={data.price_eur}
                                onChange={(e) => setData("price_eur", e.target.value)}
                                invalidFeedback={errors.price_eur}
                                min="0.01"
                                step="0.01"
                            />

                            <FormGroup
                                id="display_order"
                                label="Display Order"
                                type="number"
                                value={data.display_order}
                                onChange={(e) => setData("display_order", e.target.value)}
                                helpText="Optional: Lower numbers appear first"
                                invalidFeedback={errors.display_order}
                                min="0"
                            />
                        </Col>
                        <Col md="6">
                            <FormGroup
                                id="description"
                                label="Description"
                                as="textarea"
                                value={data.description}
                                onChange={(e) => setData("description", e.target.value)}
                                rows="6"
                                helpText="Describe the menu item"
                                invalidFeedback={errors.description}
                            />

                            <Switch
                                id="statusSwitch"
                                label="Active"
                                checked={data.status}
                                onChange={({ target: { checked } }) =>
                                    setData(prevData => ({ ...prevData, status: checked }))
                                }
                            />
                        </Col>
                    </Row>
                </form>
            </Modal>
        </>
    );
}



