import FormGroup from "@admin/ui/forms/FormGroup";
import Switch from "@admin/ui/forms/Switch";
import FileUploadInput from "@admin/ui/forms/FileUploadInput";
import Modal from "@admin/ui/Modal";
import WysiwygEditor from "@admin/ui/forms/WysiwygEditor";
import { useEffect, useState } from "react";
import { useFormResponse } from '@/hooks/useFormResponse';
import { useForm } from "@inertiajs/react";

export default function EditEvent({ isOpen, onClose, initialData }){
    const {data, setData, put, processing, errors} = useForm({
        title: '',
        event_date: '',
        location: '',
        photo: '',
        description:'',
        status: true,
    });

    const today = new Date().toISOString().slice(0, 16);

    const handleSuccess = (filename) => {
        setData('photo',filename);
    };

     const handleError = (error) => {
        console.error("Upload error:", error);
    };

    useEffect(() => {
        if (initialData) {
          setData({
              title: initialData?.title || '',
              event_date: initialData?.event_date || '',
              location: initialData?.location || '',
              photo: initialData?.photo || '',
              description: initialData?.description || '',
              status: initialData?.status === 1,
          });
        }
    }, [initialData, setData]);

    const handleSave = (event) => {
        event.preventDefault();
        const formResponse = useFormResponse(() => {
            onClose?.(); 
        });
        put(`/admin/api/update_event/${initialData.id}`, {
            preserveScroll: true,
                onSuccess: formResponse.onSuccess, 
                onError: formResponse.onError, 
            });
    };
    return(
        <>
        <Modal isOpen={isOpen} onClose={onClose} title={`Edit ${initialData?.title}`} size="large"
        primaryButtonText="Save Changes"
        onPrimaryAction={handleSave}
        processing={processing}
        >
            <form onSubmit={handleSave}>
            <FormGroup label="Event Title" 
            value={data.title}
            onChange={(e) => setData('title', e.target.value)}
            invalidFeedback={errors.title}
            />

            <FormGroup
            id="event_date"
            label="Event Date and Time"
            as="input"
            type="datetime-local"
            value={data.event_date}
            min={today}  
            onChange={(e) => setData('event_date', e.target.value)}
            invalidFeedback={errors.event_date}
            helpText="Select the date and time the event is expected to happen. Past dates and times are not allowed."
            />

            <FormGroup label="Location" 
            value={data.location}
            onChange={(e) => setData('location', e.target.value)}
            helpText="Enter address for the event or virtual if its an online event"
            invalidFeedback={errors.location}
            />

            <br />
            <label className="label">Description (Required)</label>
            <WysiwygEditor
            value={data.description}
            onChange={description => setData('description', description)}
            height={400}
            placeholder="Write your description  here..."
            toolbar="full" />
            <br />
            <FileUploadInput label="Upload image (600x400)px" 
            buttonText={"Choose file (600x400)"}
            uploadUrl="/admin/api/event/upload_image"
            onSuccess={handleSuccess}
            previewUrl={`/storage/${data.photo}`}
            onError={handleError}/>
            <Switch
            id="customSwitch"
            label="Active"
            checked={data.status}
            onChange={({ target: { checked } }) =>
                setData(prevData => ({ ...prevData, status: checked }))
                }                  
                />
                <br/>
            </form>

        </Modal>
        </>
    );
}