<?php

namespace App\Http\Controllers;

use App\Models\MenuItem;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use App\Traits\DeleteModelTrait;

class MenuItemController extends Controller
{
    use DeleteModelTrait;

    /**
     * Display a listing of the resource (public frontend).
     */
    public function index()
    {
        $menuItems = MenuItem::where('status', true)
            ->orderBy('display_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();

        // Group by category
        $grouped = [
            'Food' => [],
            'Drinks' => [],
            'Desserts' => [],
        ];

        foreach ($menuItems as $item) {
            $category = $item->category ?? null;
            if ($category && isset($grouped[$category])) {
                $grouped[$category][] = [
                    'id' => $item->id,
                    'name' => $item->name,
                    'category' => $item->category,
                    'price' => $item->price_usd ?? $item->price_fbu ?? $item->price_eur,
                    'price_fbu' => $item->price_fbu,
                    'price_usd' => $item->price_usd,
                    'price_eur' => $item->price_eur,
                    'description' => $item->description,
                ];
            }
        }

        return Inertia::render('Dining', [
            'foodItems' => $grouped['Food'],
            'drinkItems' => $grouped['Drinks'],
            'dessertItems' => $grouped['Desserts'],
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'category' => 'required|in:Food,Drinks,Desserts',
                'price_fbu' => 'required_without_all:price_usd,price_eur|nullable|numeric|min:0.01',
                'price_usd' => 'required_without_all:price_fbu,price_eur|nullable|numeric|min:0.01',
                'price_eur' => 'required_without_all:price_fbu,price_usd|nullable|numeric|min:0.01',
                'description' => 'nullable|string',
                'display_order' => 'nullable|integer|min:0',
                'status' => 'required|boolean',
            ]);

            $menuItem = new MenuItem;
            $menuItem->name = $validatedData['name'];
            $menuItem->category = $validatedData['category'];
            $menuItem->price_fbu = $validatedData['price_fbu'] ?? null;
            $menuItem->price_usd = $validatedData['price_usd'] ?? null;
            $menuItem->price_eur = $validatedData['price_eur'] ?? null;
            // Maintain legacy price column for compatibility
            $menuItem->price = $validatedData['price_usd'] ?? $validatedData['price_fbu'] ?? $validatedData['price_eur'] ?? 0;
            $menuItem->description = $validatedData['description'] ?? null;
            $menuItem->display_order = $validatedData['display_order'] ?? null;
            $menuItem->status = $validatedData['status'];
            $menuItem->save();

            DB::commit();
            return redirect()->route('admin.menus')->with('success', [
                'message' => 'Menu item created successfully',
                'check' => false
            ]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', [
                'message' => 'An error occurred: ' . $e->getMessage(),
                'check' => false
            ]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $menuItem = MenuItem::findOrFail($id);
        return response()->json([
            'id' => $menuItem->id,
            'name' => $menuItem->name,
            'category' => $menuItem->category,
            'price_fbu' => $menuItem->price_fbu,
            'price_usd' => $menuItem->price_usd,
            'price_eur' => $menuItem->price_eur,
            'description' => $menuItem->description,
            'display_order' => $menuItem->display_order,
            'status' => $menuItem->status,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'category' => 'required|in:Food,Drinks,Desserts',
                'price_fbu' => 'required_without_all:price_usd,price_eur|nullable|numeric|min:0.01',
                'price_usd' => 'required_without_all:price_fbu,price_eur|nullable|numeric|min:0.01',
                'price_eur' => 'required_without_all:price_fbu,price_usd|nullable|numeric|min:0.01',
                'description' => 'nullable|string',
                'display_order' => 'nullable|integer|min:0',
                'status' => 'required|boolean',
            ]);

            $menuItem = MenuItem::findOrFail($id);
            $menuItem->fill($validatedData);
            $menuItem->price = $validatedData['price_usd'] ?? $validatedData['price_fbu'] ?? $validatedData['price_eur'] ?? 0;
            if ($menuItem->isDirty()) {
                $menuItem->save();
            }

            DB::commit();

            return Redirect::back()->with('success', [
                'message' => 'Menu item updated successfully',
                'check' => false
            ]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return Redirect::back()->with('error', [
                'message' => 'Error: ' . $e->getMessage(),
                'check' => $e->getMessage()
            ]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        return $this->deleteModel($id, new MenuItem());
    }

    public function fetchMenuItems()
    {
        $menuItems = MenuItem::orderBy('display_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();

        // Group by category
        $grouped = [
            'Food' => [],
            'Drinks' => [],
            'Desserts' => [],
        ];

        foreach ($menuItems as $item) {
            $category = $item->category ?? null;
            if ($category && isset($grouped[$category])) {
                $grouped[$category][] = [
                    'id' => $item->id,
                    'name' => $item->name,
                    'category' => $item->category,
                    'price' => $item->price_usd ?? $item->price_fbu ?? $item->price_eur,
                    'price_fbu' => $item->price_fbu,
                    'price_usd' => $item->price_usd,
                    'price_eur' => $item->price_eur,
                    'description' => $item->description,
                    'display_order' => $item->display_order,
                    'status' => $item->status, // Pass directly as boolean, same as other controllers
                    'created_at' => $item->created_at,
                ];
            }
        }

        return Inertia::render('menus/Menu', [
            'foodItems' => $grouped['Food'],
            'drinkItems' => $grouped['Drinks'],
            'dessertItems' => $grouped['Desserts'],
        ]);
    }
}
