<?php

namespace App\Http\Controllers;

use App\Models\Event;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Redirect;
use Inertia\Inertia;
use App\Traits\DeleteModelTrait; 

class EventController extends Controller
{
    use DeleteModelTrait;
    /**
     * Display a listing of events.
     */
    public function index()
    {
        // Get event rooms for display on the meetings & events page
        $eventRooms = \App\Models\EventRoom::where('status', true)
            ->orderBy('name')
            ->get()
            ->map(function ($room) {
                return [
                    'id' => $room->id,
                    'name' => $room->name,
                    'primary_photo' => $room->primary_photo,
                    'price_fbu' => $room->price_fbu,
                    'price_usd' => $room->price_usd,
                    'price_eur' => $room->price_eur,
                    'short_description' => $room->short_description,
                    'full_description' => $room->full_description,
                    'capacity' => $room->capacity,
                    'layouts' => $room->layouts,
                    'equipment' => $room->equipment,
                    'gallery' => $room->gallery,
                ];
            });

        return Inertia::render('Event', [
            'eventRooms' => $eventRooms,
        ]);
    }

    /**
     * Store a new event.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $validatedData = $request->validate([
                'title'       => 'required|min:3|max:250',
                'photo'       => 'nullable',
                'description' => 'nullable',
                'location'    => 'nullable|string',
                'event_date'  => 'required|date|after_or_equal:' . now(),
                'status'      => 'required|boolean',
            ]);

            $event = new Event($validatedData);
            $event->save();

            DB::commit();
            return to_route('admin.events')->with('success', [
                'message' => 'Event created successfully',
                'check' => false
            ]
        );
        } catch (ValidationException $e) {
            DB::rollBack();
            return redirect()->back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return Redirect::back()->with('error', 'Error: ' . $e->getMessage());
        }
    }

    /**
     * Show the specified event.
     */
    public function show($id)
    {
        $event = Event::findOrFail($id);

        return Inertia::render('EventDetail', [
            'event' => $event
        ]);
    }

    /**
     * Update the specified event.
     */
    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();

            $validatedData = $request->validate([
                'title'       => 'required|min:3|max:250',
                'photo'       => 'nullable',
                'description' => 'nullable',
                'location'    => 'nullable|string',
                'event_date'  => 'required|date|after_or_equal:' . now(),
                'status'      => 'required|boolean',
            ]);

            $event = Event::findOrFail($id);
            $event->update($validatedData);

            DB::commit();
            return Redirect::back()->with('success', [
                'message' => 'Event updated successfully',
                'check'   => false
            ]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return Redirect::back()->with('error', [
                'message' => 'Validation error: ' . $e->getMessage(),
                'check'   => $e->getMessage()
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return Redirect::back()->with('error', [
                'message' => 'Validation error: ' . $e->getMessage(),
                'check'   => $e->getMessage()
            ]);
        }
    }

    public function edit($id)
    {
        //
        $event = Event::findOrFail($id);
        return response()->json($event);
    }

    /**
     * Remove the specified event from storage.
     */
    public function destroy($id)
    {
        return $this->deleteModel($id, new Event());

    }

    public function uploadImage(Request $request)
    {
        $request->validate([
            'file' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:5048',
        ]);

        $path = $request->file('file')->store('events/images', 'public');
        $filename = basename($path);

        return response()->json(['filename' => 'events/images/' . $filename]);
    }

    public function fetchEvents()
    {
        $events = Event::orderBy('created_at', 'desc')->get();
        return Inertia::render('events/Event', [
            'events' => $events->map(function ($event) {
                return $event->only('id','title', 'photo' ,'status','created_at');
            }),
        ]);
    }



}
